<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\Reflection\CallableDeclaration;

use Magento\Mray\CodeStructuralElement\Php\Reflection\MethodDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\TypeFactory;
use phpDocumentor\Reflection\Type;
use PHPStan\PhpDocParser\Ast\PhpDoc\MethodTagValueNode;
use PHPStan\PhpDocParser\Ast\PhpDoc\MethodTagValueParameterNode;
use function array_map;
use function sha1;
use function sprintf;

class PhpDocMethodTag implements MethodDeclaration
{
    /** @var MethodTagValueNode  */
    private $methodTagValueNode;
    /** @var string|null  */
    private $classLikeName;

    /**
     * @param MethodTagValueNode $methodTagValueNode
     * @param string|null $classLikeName
     */
    public function __construct(
        MethodTagValueNode $methodTagValueNode,
        ?string $classLikeName
    ) {
        $this->methodTagValueNode = $methodTagValueNode;
        $this->classLikeName = $classLikeName;
    }

    /**
     * @return string|null
     */
    public function getName(): ?string
    {
        return $this->methodTagValueNode->methodName;
    }

    /**
     * @return string|null
     */
    public function getClassLikeName(): ?string
    {
        return $this->classLikeName;
    }

    /**
     * @inheritDoc
     */
    public function getParams(): array
    {
        $params = array_map(function (MethodTagValueParameterNode $paramTag) {
            return new PhpDocMethodTagValueParameter($paramTag);
        }, $this->methodTagValueNode->parameters);
        return $params;
    }

    /**
     * @return Type
     */
    public function getReturnType(): Type
    {
        return TypeFactory::cast($this->methodTagValueNode->returnType);
    }

    /**
     * @return bool
     */
    public function isPublic(): bool
    {
        return true;
    }

    /**
     * @return bool
     */
    public function isProtected(): bool
    {
        return false;
    }

    /**
     * @return bool
     */
    public function isPrivate(): bool
    {
        return false;
    }

    /**
     * @return bool
     */
    public function isAbstract(): bool
    {
        return false;
    }

    /**
     * @return bool
     */
    public function isFinal(): bool
    {
        return false;
    }

    /**
     * @return bool
     */
    public function isStatic(): bool
    {
        return $this->methodTagValueNode->isStatic;
    }

    /**
     * @return bool
     */
    public function isMagic(): bool
    {
        return false;
    }

    /**
     * @return Type
     */
    public function getValueType(): Type
    {
        return $this->getReturnType();
    }

    /**
     * @return string|null
     */
    public function getImplementationChecksum(): ?string
    {
        return null;
    }
}
