<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeDeclaration;

use Magento\Mray\CodeStructuralElement\Php\NodeVisitor\PhpDocAst;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeMemberDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ConstantDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\Context\StructuralElementDeclarations;
use Magento\Mray\CodeStructuralElement\Php\Reflection\CallableDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\InterfaceDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\MethodDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\InCodeDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\PropertyDeclaration;
use PhpParser\Node\Stmt;
use PhpParser\NodeDumper;
use PHPStan\PhpDocParser\Ast\PhpDoc\PhpDocNode;
use function array_map;
use function array_values;
use function get_class;
use function in_array;
use function sha1;
use function sprintf;
use function uniqid;

abstract class ClassLike implements ClassLikeDeclaration, InCodeDeclaration
{
    /**
     * @var Stmt\ClassLike
     */
    private $node;
    /**
     * @var StructuralElementDeclarations|null
     */
    private $elementsRegistry;

    /**
     * @param Stmt\ClassLike $node
     * @param StructuralElementDeclarations|null $elementsRegistry
     */
    public function __construct(Stmt\ClassLike $node, ?StructuralElementDeclarations $elementsRegistry = null)
    {
        $this->node = $node;
        $this->elementsRegistry = $elementsRegistry;
    }

    /**
     * @return string|null
     */
    final public function getName(): ?string
    {
        $n = $this->getNode();
        return isset($n->namespacedName) ? (string)$n->namespacedName : null;
    }

    /**
     * @return ClassLikeMemberDeclaration[]
     */
    final public function getMembers(): array
    {
        $members = $this->getOwnMembers();
        if (!$this->elementsRegistry) {
            return $members;
        }

        $membersIndex = array_map(function (ClassLikeMemberDeclaration $member): string {
            return $this->getMemberId($member);
        }, $members);
        foreach ($this->getImplementationProviders() as $ip) {
            if ($this->getName() === $ip) {
                continue;
            }
            foreach ($this->elementsRegistry->findClassLike($ip) as $ipd) {
                foreach ($ipd->getMembers() as $member) {
                    $memberId = $this->getMemberId($member);
                    if (!in_array($memberId, $membersIndex)) {
                        $members[] = $member;
                        $membersIndex[] = $memberId;
                    }
                }
            }
        }

        return $members;
    }

    /**
     * @return array
     */
    public function getInheritedMembers(): array
    {
        if (!$this->elementsRegistry) {
            return [];
        }

        $members = $this->getOwnMembers();
        $membersIndex = [];
        foreach ($members as $member) {
            $membersIndex[$this->getMemberId($member)] = $member;
        }
        $overridden = [];
        foreach ($this->getImplementationProviders() as $ip) {
            foreach ($this->elementsRegistry->findClassLike($ip) as $ipd) {
                foreach ($ipd->getMembers() as $member) {
                    $memberId = $this->getMemberId($member);
                    $overridden[] = [
                        'inherited' => $member,
                        'own' => $membersIndex[$memberId] ?? null,
                        'type' => $this->getMemberInheritanceType($member, $membersIndex[$memberId] ?? null)
                    ];
                }
            }
        }

        return $overridden;
    }

    /**
     * @param ClassLikeMemberDeclaration $inheritedMember
     * @param ClassLikeMemberDeclaration|null $ownMember
     * @return string
     */
    protected function getMemberInheritanceType(
        ClassLikeMemberDeclaration $inheritedMember,
        ?ClassLikeMemberDeclaration $ownMember
    ): string {
        if ($ownMember === null) {
            return 'inherit';
        }
        if (!$this instanceof ClassDeclaration) {
            return 'override';
        }

        if (!$inheritedMember instanceof MethodDeclaration) {
            return 'override';
        }

        if ($inheritedMember->isAbstract() && !$ownMember->isAbstract()) {
            return 'implement';
        }

        if (!$this->elementsRegistry) {
            return 'override';
        }

        $inheritedFrom = $this->elementsRegistry->findClassLike($inheritedMember->getClassLikeName());
        if (empty($inheritedFrom)) {
            return 'override';
        }

        foreach ($inheritedFrom as $d) {
            if (!$d instanceof InterfaceDeclaration) {
                return 'override';
            }
        }
        return 'implement';
    }

    /**
     * @param ClassLikeMemberDeclaration $member
     * @return string
     */
    private function getMemberId(ClassLikeMemberDeclaration $member): string
    {

        return sprintf('%s(%s)', get_class($member), $member->getName() ?: uniqid());
    }

    /**
     * @var array
     */
    private $ownMembers;
    /**
     * @return ClassLikeMemberDeclaration[]
     */
    final public function getOwnMembers(): array
    {
        if (isset($this->ownMembers)) {
            return array_values($this->ownMembers);
        }

        $this->ownMembers = [];

        $phpDoc = $this->getPhpDoc();
        if ($phpDoc) {
            foreach ($phpDoc->getPropertyTagValues() as $prop) {
                $this->ownMembers[] = new PropertyDeclaration\PhpDocPropertyTag(
                    $prop,
                    $this->getName()
                );
            }
            foreach ($phpDoc->getPropertyReadTagValues() as $prop) {
                $this->ownMembers[] = new PropertyDeclaration\PhpDocPropertyTag(
                    $prop,
                    $this->getName()
                );
            }
            foreach ($phpDoc->getPropertyWriteTagValues() as $prop) {
                $this->ownMembers[] = new PropertyDeclaration\PhpDocPropertyTag(
                    $prop,
                    $this->getName()
                );
            }
            foreach ($phpDoc->getMethodTagValues() as $method) {
                $this->ownMembers[] = new CallableDeclaration\PhpDocMethodTag(
                    $method,
                    $this->getName()
                );
            }
        }

        foreach ($this->node->getConstants() as $constsGroupNode) {
            foreach ($constsGroupNode->consts as $constNode) {
                $this->ownMembers[] = new ConstantDeclaration\ClassConst(
                    $constsGroupNode,
                    $constNode,
                    $this->getName(),
                    $this->getInheritedMemberFinder(ConstantDeclaration::class)
                );
            }
        }
        foreach ($this->node->getProperties() as $propsGroupNode) {
            foreach ($propsGroupNode->props as $propNode) {
                $this->ownMembers[] = new PropertyDeclaration\Property(
                    $propsGroupNode,
                    $propNode,
                    $this->getName(),
                    $this->getInheritedMemberFinder(PropertyDeclaration::class)
                );
            }
        }
        foreach ($this->node->getMethods() as $methodNode) {
            $this->ownMembers[] = new CallableDeclaration\Method(
                $methodNode,
                $this->getName(),
                $this->getInheritedMemberFinder(MethodDeclaration::class)
            );
        }

        return array_values($this->ownMembers);
    }

    /** @var array  */
    private $inheritedMemberFinders;

    /**
     * @param string $memberInterface
     * @return Finder|null
     */
    private function getInheritedMemberFinder(string $memberInterface): ?Finder
    {
        if (isset($this->inheritedMemberFinders[$memberInterface])) {
            return $this->inheritedMemberFinders[$memberInterface];
        }
        if (!$this->elementsRegistry) {
            return null;
        }
        $this->inheritedMemberFinders[$memberInterface] = new Finder(
            $memberInterface,
            $this->elementsRegistry,
            $this->getImplementationProviders()
        );

        return $this->inheritedMemberFinders[$memberInterface];
    }

    /**
     * @return string|null
     */
    public function getImplementationChecksum(): ?string
    {
        $dumper = new NodeDumper();
        return sprintf('v1:sha1:%s', sha1($dumper->dump($this->node)));
    }

    /**
     * @inheritDoc
     */
    public function getDeclarationNodes(): array
    {
        return [
            $this->node
        ];
    }

    /**
     * @return PhpDocNode|null
     */
    public function getPhpDoc(): ?PhpDocNode
    {
        return $this->node->getAttribute(PhpDocAst::PHP_DOC_COMMENT);
    }
}
