<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\Reflection;

use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeDeclaration\Finder;
use PHPStan\PhpDocParser\Ast\PhpDoc\PhpDocNode;
use PHPStan\PhpDocParser\Ast\PhpDoc\PhpDocTagNode;
use PHPStan\PhpDocParser\Ast\PhpDoc\PhpDocTextNode;
use function array_push;
use function array_reverse;
use function strcasecmp;

trait ClassLikeMemberPhpDocInheritance
{
    /**
     * @var Finder|null
     */
    private $inheritedDeclarationsFinder;

    /**
     * @param PhpDocNode|null $phpDoc
     * @return PhpDocNode|null
     */
    private function getPhpDocWithResolvedInheritance(?PhpDocNode $phpDoc): ?PhpDocNode
    {
        if (!$phpDoc || !$this->isInheritDoc($phpDoc)) {
            return $phpDoc;
        }

        $mergedDocs = [];
        foreach (array_reverse($this->findInheritedDeclarations()) as $inheritedDeclaration) {
            if ($inheritedDeclaration instanceof self) {
                $inheritedDoc = $inheritedDeclaration->getPhpDoc();
                if ($inheritedDoc) {
                    array_push($mergedDocs, ...$inheritedDoc->children);
                }
            }
        }

        if ($mergedDocs) {
            array_push($mergedDocs, ...$phpDoc->children);
            return new PhpDocNode($mergedDocs);
        } else {
            return $phpDoc;
        }
    }
    /**
     * @return ClassLikeMemberDeclaration[]
     */
    private function findInheritedDeclarations(): array
    {
        $finder = $this->inheritedDeclarationsFinder;
        if (!$finder) {
            return [];
        }
        return $finder->getInheritedDeclarations($this->getName());
    }

    /**
     * @param PhpDocNode $phpDoc
     * @return bool
     */
    private function isInheritDoc(PhpDocNode $phpDoc): bool
    {
        foreach ($phpDoc->children as $docNode) {
            if ($docNode instanceof PhpDocTagNode && strcasecmp($docNode->name, '@inheritDoc') === 0) {
                return true;
            } elseif ($docNode instanceof PhpDocTextNode && strcasecmp($docNode->text, '{@inheritDoc}') === 0) {
                return true;
            }
        }
        return false;
    }
}
