<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\Reflection\ConstantDeclaration;

use Magento\Mray\CodeStructuralElement\Php\NodeRuntimeTypeResolver\Memorized;
use Magento\Mray\CodeStructuralElement\Php\NodeRuntimeTypeResolver\Resolver;
use Magento\Mray\CodeStructuralElement\Php\NodeVisitor\PhpDocAst;
use Magento\Mray\CodeStructuralElement\Php\Reflection\GlobalConstantDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\InCodeDeclaration;
use phpDocumentor\Reflection\Type;
use PhpParser\Node;
use PhpParser\NodeDumper;
use PHPStan\PhpDocParser\Ast\PhpDoc\PhpDocNode;
use function array_values;
use function sha1;
use function sprintf;

class Const_ implements GlobalConstantDeclaration, InCodeDeclaration
{
    /** @var Node\Stmt\Const_ */
    private $group;
    /** @var Node\Const_ */
    private $const;

    /** @var Memorized|Resolver */
    private $typeResolver;

    /**
     * @param Node\Stmt\Const_ $group
     * @param Node\Const_ $const
     */
    public function __construct(
        Node\Stmt\Const_ $group,
        Node\Const_ $const
    ) {
        $this->group = $group;
        $this->const = $const;

        $this->typeResolver = new Memorized();
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return (string)$this->const->name;
    }

    /**
     * @return Type
     */
    public function getType(): Type
    {
        return $this->typeResolver->resolve($this->const->value);
    }

    /**
     * @return Type
     */
    public function getValueType(): Type
    {
        return $this->getType();
    }

    /**
     * @return string|null
     */
    public function getImplementationChecksum(): ?string
    {
//        return null;
        $g = clone $this->group;
        foreach ($g->consts as $key => $const) {
            if ($const !== $this->const) {
                unset($g->consts[$key]);
            }
        }
        $g->consts = array_values($g->consts);

        $dumper = new NodeDumper();
        return sprintf('v1:sha1:%s', sha1($dumper->dump($g)));
    }

    /**
     * @inheritDoc
     */
    public function getDeclarationNodes(): array
    {
        return [
            $this->group,
            $this->const,
        ];
    }

    /**
     * @return PhpDocNode|null
     */
    public function getPhpDoc(): ?PhpDocNode
    {
        return $this->group->getAttribute(PhpDocAst::PHP_DOC_COMMENT);
    }
}
