<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\Reflection\Context;

use InvalidArgumentException;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\Context;
use Magento\Mray\CodeStructuralElement\Php\Reflection\VariableDefinition;
use PhpParser\Node;
use function get_class;
use function sprintf;

class ClassLike implements Context
{
    use VariableScope;

    /**
     * @var Global_
     */
    private $global;
    /**
     * @var VariableDefinition\ThisPreDefinition
     */
    private $thisPreDefinition;

    /**
     * @param Node\Stmt\ClassLike $node
     * @param Global_ $global
     */
    public function __construct(Node\Stmt\ClassLike $node, Global_ $global)
    {
        $this->global = $global;
        $this->thisPreDefinition = new VariableDefinition\ThisPreDefinition($this->createClassLikeDeclaration(
            $node,
            $global->getStructuralElementDeclarations()
        ));
    }

    /**
     * @return VariableDefinition\ThisPreDefinition
     */
    public function getThisVariableDefinition(): VariableDefinition\ThisPreDefinition
    {
        return $this->thisPreDefinition;
    }

    /**
     * @return Global_
     */
    public function getGlobal(): Global_
    {
        return $this->global;
    }

    /**
     * @param Node\Stmt\ClassLike $node
     * @param StructuralElementDeclarations|null $elementsRegistry
     * @return ClassLikeDeclaration
     */
    private function createClassLikeDeclaration(
        Node\Stmt\ClassLike $node,
        ?StructuralElementDeclarations $elementsRegistry
    ): ClassLikeDeclaration {
        if ($node instanceof Node\Stmt\Class_) {
            return new ClassLikeDeclaration\Class_($node, $elementsRegistry);
        } elseif ($node instanceof Node\Stmt\Interface_) {
            return new ClassLikeDeclaration\Interface_($node, $elementsRegistry);
        } elseif ($node instanceof Node\Stmt\Trait_) {
            return new ClassLikeDeclaration\Trait_($node, $elementsRegistry);
        } else {
            throw new InvalidArgumentException(sprintf('Support of %s is not implemented yet.', get_class($node)));
        }
    }
}
