<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\Reflection\PropertyDeclaration;

use Magento\Mray\CodeStructuralElement\Php\Reflection\PropertyDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\TypeFactory;
use phpDocumentor\Reflection\Type;
use PHPStan\PhpDocParser\Ast\PhpDoc\PropertyTagValueNode;
use function ltrim;
use function sha1;
use function sprintf;

class PhpDocPropertyTag implements PropertyDeclaration
{
    /** @var PropertyTagValueNode  */
    private $pt;
    /** @var string|null  */
    private $classLikeName;

    /**
     * @param PropertyTagValueNode $pt
     * @param string|null $classLikeName
     */
    public function __construct(
        PropertyTagValueNode $pt,
        ?string $classLikeName
    ) {
        $this->pt = $pt;
        $this->classLikeName = $classLikeName;
    }

    /**
     * @return string|null
     */
    public function getName(): ?string
    {
        return ltrim($this->pt->propertyName, '$') ?: null;
    }

    /**
     * @return string|null
     */
    public function getClassLikeName(): ?string
    {
        return $this->classLikeName;
    }

    /**
     * @return Type
     */
    public function getType(): Type
    {
        return TypeFactory::cast($this->pt->type);
    }

    /**
     * @return Type
     */
    public function getValueType(): Type
    {
        return $this->getType();
    }

    /**
     * @return bool
     */
    public function isPublic(): bool
    {
        return true;
    }

    /**
     * @return bool
     */
    public function isProtected(): bool
    {
        return false;
    }

    /**
     * @return bool
     */
    public function isPrivate(): bool
    {
        return false;
    }

    /**
     * @return bool
     */
    public function isStatic(): bool
    {
        return false;
    }

    /**
     * @return string|null
     */
    public function getImplementationChecksum(): ?string
    {
        return null;
        // phpcs:ignore Squiz.PHP.NonExecutableCode
        return sprintf('v1:sha1:%s', sha1($this->pt->__toString()));
    }
}
