<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\Reflection;

use Exception;
use Magento\Mray\CodeStructuralElement\Php\Reflection\Type\CodeDiscoveryDependentType;
use phpDocumentor\Reflection\Type;
use phpDocumentor\Reflection\TypeResolver;
use phpDocumentor\Reflection\Types;
use phpDocumentor\Reflection\Types\Context;
use PhpParser\Node;
use PHPStan\PhpDocParser\Ast\Type\ArrayShapeNode;
use PHPStan\PhpDocParser\Ast\Type\TypeNode;
use function array_map;
use function is_array;
use function sprintf;
use function str_replace;
use function substr;

class TypeFactory
{
    /** @var array  */
    private static $pool = [];

    /**
     * @param string $type
     * @param mixed $typeArgs
     * @return Type
     */
    public static function get(string $type, ...$typeArgs): Type
    {
        $instanceRefKey = self::getInstanceRefKey($type, $typeArgs);
        if (isset(self::$pool[$instanceRefKey])) {
            return self::$pool[$instanceRefKey];
        }

        $instance = self::instantiate($type, $typeArgs);
        self::$pool[$instanceRefKey] = $instance;
        return $instance;
    }

    /**
     * @param string $type
     * @param array $typeArgs
     * @return Type
     */
    private static function instantiate(string $type, array $typeArgs): Type
    {
        return new $type(...$typeArgs);
    }

    /**
     * @param string $type
     * @param array $typeArgs
     * @return string|null
     */
    private static function getInstanceRefKey(string $type, array $typeArgs): ?string
    {
        if (!$typeArgs) {
            return $type;
        }

        return sprintf('%s(%s)', $type, self::typeArgsKey($typeArgs));
    }

    /**
     * @param array $parts
     * @return string
     */
    private static function typeArgsKey(array $parts): string
    {
        static $delimiter = ', ';

        if (!$parts) {
            return '';
        }

        $key = '';
        foreach ($parts as $part) {
            if (is_array($part)) {
                $key .= '[' . self::typeArgsKey($part) . ']';
            } else {
                $key .= $part;
            }
            $key .= $delimiter;
        }

        return substr($key, 0, -2);
    }

    /**
     * @param mixed $type
     * @param Context|null $context
     * @return Type
     */
    public static function cast($type, ?Context $context = null): Type
    {
        if (!isset($type)) {
            return TypeFactory::get(Types\Mixed_::class);
        }

        static $typeResolver;
        if (!isset($typeResolver)) {
            $typeResolver = new TypeResolver();
        }

        if ($type instanceof Node\NullableType) {
            return TypeFactory::get(Types\Compound::class, [
                $typeResolver->resolve((string)$type->type, $context),
                TypeFactory::get(Types\Null_::class),
            ]);
        }
        if ($type instanceof Node\UnionType) {
            return TypeFactory::get(Types\Compound::class, array_map(function ($type) use ($typeResolver, $context) {
                return $typeResolver->resolve((string)$type, $context);
            }, $type->types));
        }

        $typeSignature = (string)$type;
        if ($type instanceof TypeNode) {
            $typeSignature = str_replace([' ', '&'], ['', '|'], $typeSignature);
        }

        try {
            return $typeResolver->resolve($typeSignature, $context);
        } catch (Exception $e) {
            if ($type instanceof ArrayShapeNode) {
                return TypeFactory::get(Types\Array_::class);
            }
            return TypeFactory::get(Types\Mixed_::class);
        }
    }

    /**
     * @param Type $t1
     * @param Type $t2
     * @return Type
     */
    public static function preciseType(Type $t1, Type $t2): Type
    {
        // todo: implement flexible types merge strategy
        if (self::firstHasLessPreciseType($t1, $t2)) {
            return $t2;
        }
        return $t1;
    }

    /**
     * @param Type $t1
     * @param Type $t2
     * @return bool
     */
    private static function firstHasLessPreciseType(Type $t1, Type $t2): bool
    {
        return self::firstIsUnknownAndSecondHasDefinedType($t1, $t2)
            || self::firstTypeIsGenericObjectAndSecondHasDefinedType($t1, $t2)
            || self::firstTypeIsGenericArrayAndSecondIsTyped($t1, $t2);
    }

    /**
     * @param Type $t1
     * @param Type $t2
     * @return bool
     */
    private static function firstIsUnknownAndSecondHasDefinedType(Type $t1, Type $t2): bool
    {
        if (self::isDefinedType($t1)) {
            return false;
        }
        return self::isDefinedType($t2);
    }

    /**
     * @param Type $t
     * @return bool
     */
    private static function isDefinedType(Type $t): bool
    {
        if ($t instanceof Types\Mixed_) {
            return false;
        }
        if ($t instanceof CodeDiscoveryDependentType) {
            return self::isDefinedType($t->getEstimatedType());
        }

        return true;
    }

    /**
     * @param Type $t1
     * @param Type $t2
     * @return bool
     */
    private static function firstTypeIsGenericObjectAndSecondHasDefinedType(Type $t1, Type $t2): bool
    {
        if (!$t1 instanceof Types\Object_) {
            return false;
        }
        if ($t1->getFqsen()) {
            return false;
        }
        if (!$t2 instanceof Types\Object_) {
            return false;
        }
        return (bool)$t2->getFqsen();
    }

    /**
     * @param Type $t1
     * @param Type $t2
     * @return bool
     */
    private static function firstTypeIsGenericArrayAndSecondIsTyped(Type $t1, Type $t2): bool
    {
        if (!$t1 instanceof Types\Array_) {
            return false;
        }
        if (!$t2 instanceof Types\Array_) {
            return false;
        }
        if (!$t1->getValueType() instanceof Types\Mixed_) {
            return false;
        }

        // todo: add check of key types (rare usage)
        return self::firstHasLessPreciseType($t1->getValueType(), $t2->getValueType());
    }
}
