<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\Reflection\ValueAccessor;

use Magento\Mray\CodeStructuralElement\Php\Reflection\TypeFactory;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ValueAccessor;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ValuePointer;
use phpDocumentor\Reflection\Types;
use PhpParser\Node;
use function array_merge;
use function is_scalar;
use function is_string;

class Expr implements ValueAccessor
{
    /** @var Node\Expr  */
    private $expr;
    /** @var KnownPath  */
    private $knownPath;

    /**
     * @param Node\Expr $expr
     */
    public function __construct(Node\Expr $expr)
    {
        $this->expr = $expr;
    }

    /**
     * @inheritDoc
     */
    public function getPath(): array
    {
        return $this->getKnownPath()->getPath();
    }

    /**
     * @return bool
     */
    public function resolvable(): bool
    {
        return $this->getKnownPath()->resolvable();
    }

    /**
     * @param ValueAccessor $other
     * @return bool
     */
    public function equal(ValueAccessor $other): bool
    {
        return $this->getKnownPath()->equal($other);
    }

    /**
     * @return KnownPath
     */
    private function getKnownPath(): KnownPath
    {
        if (!isset($this->knownPath)) {
            $this->knownPath = new KnownPath($this->readPathFromExpr(
                $this->expr,
                function () {
                    return TypeFactory::get(Types\Mixed_::class);
                },
                []
            ));
        }
        return $this->knownPath;
    }

    /**
     * @param Node\Expr $expr
     * @param callable $type
     * @param array $path
     * @return array
     */
    private function readPathFromExpr(Node\Expr $expr, callable $type, array $path): array
    {
        if ($expr instanceof Node\Expr\Variable) {
            if (is_string($expr->name)) {
                return array_merge([new ValuePointer($expr->name, $type())], $path);
            }
        } elseif ($expr instanceof Node\Expr\ArrayDimFetch) {
            if (isset($expr->dim->value) && is_scalar($expr->dim->value)) {
                return $this->readPathFromExpr(
                    $expr->var,
                    function () {
                        return TypeFactory::get(Types\Array_::class);
                    },
                    array_merge([new ValuePointer((string)$expr->dim->value, $type())], $path)
                );
            }
        } elseif ($expr instanceof Node\Expr\PropertyFetch) {
            if ($expr->name instanceof Node\Identifier) {
                return $this->readPathFromExpr(
                    $expr->var,
                    function () {
                        return TypeFactory::get(Types\Object_::class);
                    },
                    array_merge([new ValuePointer($expr->name->toString(), $type())], $path)
                );
            }
        }
        return [];
    }
}
