<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\Reflection\VariableDefinition;

use Magento\Mray\CodeStructuralElement\Php\NodeRuntimeTypeResolver\Memorized;
use Magento\Mray\CodeStructuralElement\Php\NodeRuntimeTypeResolver\Resolver;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ValueAccessor;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ValuePointer;
use Magento\Mray\CodeStructuralElement\Php\Reflection\VariableDefinition;
use phpDocumentor\Reflection\Type;
use PhpParser\Node\Expr;
use PhpParser\Node\Scalar;
use TypeError;
use function get_class;
use function in_array;
use function sprintf;

class ArrayAssignment implements VariableDefinition
{
    /**
     * @var Expr|Expr\Assign
     */
    private $expr;
    /**
     * @var Expr\ArrayItem
     */
    private $item;
    /**
     * @var Memorized|Resolver
     */
    private $typeResolver;
    /**
     * @var ValueAccessor
     */
    private $accessor;

    /**
     * @param Expr $expr
     * @param Expr\ArrayItem $item
     */
    public function __construct(Expr $expr, Expr\ArrayItem $item)
    {
        if (!($expr instanceof Expr\Assign)) {
            throw new TypeError(sprintf('Expression is not an assignment (instance of %s given).', get_class($expr)));
        }
        if (!($expr->expr instanceof Expr\Array_)) {
            throw new TypeError(sprintf(
                'Expression is not an array assignment (expr is instance of %s).',
                get_class($expr->expr)
            ));
        }
        if (!in_array($item, $expr->expr->items)) {
            throw new TypeError("Provided item is not defined in array.");
        }

        $this->expr = $expr;
        $this->item = $item;
        $this->typeResolver = new Memorized();
    }

    /**
     * @return bool
     */
    public function isDefinition(): bool
    {
        return true;
    }

    /**
     * @return ValueAccessor
     */
    public function getAccessor(): ValueAccessor
    {
        if (!isset($this->accessor)) {
            $this->accessor = $this->createAccessor();
        }
        return $this->accessor;
    }

    /**
     * @return ValueAccessor
     */
    private function createAccessor(): ValueAccessor
    {
        $key = null;
        if (!$this->item->key instanceof Scalar || !isset($this->item->key->value)) {
            return new ValueAccessor\Unresolvable();
        }
        $accessor = new ValueAccessor\Expr($this->expr->var);
        if (!$accessor->resolvable()) {
            return new ValueAccessor\Unresolvable();
        }
        $accessPath = $accessor->getPath();
        $accessPath[] = new ValuePointer(
            (string)$this->item->key->value,
            $this->getType()
        );
        return new ValueAccessor\KnownPath($accessPath);
    }

    /**
     * @return Type
     */
    public function getType(): Type
    {
        return $this->typeResolver->resolve($this->item);
    }
}
