<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\Reflection\VariableDefinition;

use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\PropertyDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\TypeFactory;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ValueAccessor;
use Magento\Mray\CodeStructuralElement\Php\Reflection\VariableDefinition;
use phpDocumentor\Reflection\Fqsen;
use phpDocumentor\Reflection\Type;
use phpDocumentor\Reflection\Types;
use function array_map;
use function ltrim;

class ThisPreDefinition implements VariableDefinition, ThisDefinition
{
    /** @var ClassLikeDeclaration  */
    private $declaration;
    /** @var ValueAccessor  */
    private $valueAccessor;

    /**
     * @param ClassLikeDeclaration $declaration
     */
    public function __construct(ClassLikeDeclaration $declaration)
    {
        $this->declaration = $declaration;
    }

    /**
     * @return bool
     */
    public function isDefinition(): bool
    {
        return false;
    }

    /**
     * @return ValueAccessor
     */
    public function getAccessor(): ValueAccessor
    {
        if (!isset($this->valueAccessor)) {
            $this->valueAccessor = new ValueAccessor\VariableName('this', $this->getType());
        }
        return $this->valueAccessor;
    }

    /**
     * @return Type
     */
    public function getType(): Type
    {
        $name = $this->declaration->getName();
        if ($name) {
            return $this->castNameToType($name);
        }

        // anonymous classes
        $parents = array_map(function (string $name): Type {
            return $this->castNameToType($name);
        }, $this->declaration->getImplementationProviders());

        if (!empty($parents)) {
            return TypeFactory::get(Types\Compound::class, $parents);
        }

        return TypeFactory::get(Types\Object_::class);
    }

    /**
     * @param string $name
     * @return Type
     */
    private function castNameToType(string $name): Type
    {
        $fqsen = '\\' . ltrim((string)$name, '\\');
        return TypeFactory::get(Types\Object_::class, new Fqsen($fqsen));
    }

    /**
     * @return Type|null
     */
    public function getParentType(): ?Type
    {
        if (!$this->declaration instanceof ClassDeclaration) {
            return null;
        }
        if ($this->declaration->getParentClass()) {
            return TypeFactory::get(Types\Object_::class, new Fqsen('\\' . $this->declaration->getParentClass()));
        }
        return null;
    }

    /**
     * @return array
     */
    public function getPropertiesDefinition(): array
    {
        $propertyDefinitions = [];
        $class = $this->getType();

        foreach ($this->declaration->getMembers() as $member) {
            if ($member instanceof PropertyDeclaration) {
                $propertyDefinitions[] = new DeclaredProperty($member, $class);
            }
        }

        return $propertyDefinitions;
    }
}
