<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\Usage;

use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeMemberDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\PropertyDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\Type\ClassLikeMemberValueType;
use Magento\Mray\CodeStructuralElement\Php\Reflection\Type\CodeDiscoveryDependentType;
use phpDocumentor\Reflection\Type;
use phpDocumentor\Reflection\Types;
use PhpParser\Node;
use function array_filter;
use function in_array;
use function iterator_to_array;
use function ltrim;
use function sprintf;

trait ClassMemberAccess
{
    /**
     * @param Node $node
     * @param string $usageType
     * @return array
     */
    public function checkFromRuntimeType(Node $node, string $usageType): array
    {
        if (!isset($node->runtimeType) || !$node->runtimeType instanceof ClassLikeMemberValueType) {
            return [];
        }

        $usages = [];
        $details = $this->getBasicDetails($node);
        $directClasses = array_filter(array_map(
            function (Type $type) {
                if ($type instanceof Types\Object_ && $type->getFqsen()) {
                    return ltrim((string)$type->getFqsen(), '\\');
                } else {
                    return null;
                }
            },
            iterator_to_array(
                $this->simpleTypes($node->runtimeType->getClassLikeType()),
                false
            )
        ));
        foreach ($node->runtimeType->getAccessedMembers() as $m) {
            $usages[] = new UsageCase(
                sprintf(
                    '%s::%s%s',
                    $m->getClassLikeName(),
                    $m instanceof PropertyDeclaration ? '$' : '',
                    $m->getName()
                ),
                $usageType,
                [
                    'position' => [
                        'startLine' => $node->getStartLine(),
                        'endLine' => $node->getEndLine(),
                        'startFilePos' => $node->getStartFilePos(),
                        'endFilePos' => $node->getEndFilePos(),
                    ],
                ],
                // phpcs:ignore Magento2.Performance.ForeachArrayMerge
                array_merge(
                    $details,
                    $this->usedMemberDetails($directClasses, $m)
                )
            );
        }
        return $usages;
    }

    /**
     * @param Node $node
     * @param string $usageType
     * @param array $position
     * @return array|UsageCase[]
     */
    public function checkFromNode(Node $node, string $usageType, array $position = []): array
    {
        $memberName = $node->name ?? null;
        if (!$memberName || $memberName instanceof Node\Expr) {
            return [];
        }

        if ($node instanceof Node\Expr\PropertyFetch ||
            $node instanceof Node\Expr\NullsafePropertyFetch ||
            $node instanceof Node\Expr\StaticPropertyFetch
        ) {
            $memberName = '$' . $memberName;
        } else {
            $memberName = (string)$memberName;
        }

        if (isset($node->var)) {
            return $this->usages(
                $node->var->runtimeType ?? null,
                $memberName,
                $usageType,
                $position ?: $this->getPosition($node),
                $this->getBasicDetails($node)
            );
        } elseif (isset($node->class)) {
            if (!$node->class instanceof Node\Name) {
                return [];
            }

            if (!$node->class->isSpecialClassName()) {
                return [new UsageCase(
                    $node->class . '::' . $memberName,
                    $usageType,
                    [
                        'position' => $position ?: $this->getPosition($node),
                    ],
                    $this->getBasicDetails($node)
                )];
            }

            // if we are here then special class name is used and we dont know declaration of accessed member
            if (isset($node->runtimeType) && $node->runtimeType instanceof ClassLikeMemberValueType) {
                return $this->usages(
                    $node->runtimeType->getClassLikeType(),
                    $memberName,
                    $usageType,
                    $position ?: $this->getPosition($node),
                    $this->getBasicDetails($node)
                );
            }

            return [];
        } else {
            return [];
        }
    }

    /**
     * @param Node $node
     * @return array
     */
    private function getPosition(Node $node)
    {
        return [
            'startLine' => $node->getStartLine(),
            'endLine' => $node->getEndLine(),
            'startFilePos' => $node->getStartFilePos(),
            'endFilePos' => $node->getEndFilePos(),
        ];
    }

    /**
     * @param Type|null $t
     * @param string $member
     * @param string $usageType
     * @param array $position
     * @param array $details
     * @return array
     */
    private function usages(?Type $t, string $member, string $usageType, array $position, array $details = [])
    {
        $usages = [];
        foreach ($this->simpleTypes($t) as $t) {
            if ($t instanceof Types\Object_ && $t->getFqsen()) {
                $usages[] = new UsageCase(
                    sprintf('%s::%s', ltrim((string)$t->getFqsen(), '\\'), $member),
                    $usageType,
                    [
                        'position' => $position,
                    ],
                    $details
                );
            }
        }

        return $usages;
    }

    /**
     * @param Type $t
     * @return Type[]
     */
    private function simpleTypes(?Type $t): iterable
    {
        if ($t === null) {
            return;
            // phpcs:ignore Squiz.PHP.NonExecutableCode
            yield;
        }

        if ($t instanceof CodeDiscoveryDependentType) {
            yield from $this->simpleTypes($t->getEstimatedType());
            return;
        } elseif ($t instanceof Types\Expression) {
            yield from $this->simpleTypes($t->getValueType());
            return;
        } elseif ($t instanceof Types\AggregatedType) {
            foreach ($t->getIterator() as $tt) {
                yield from $this->simpleTypes($tt);
            }
            return;
            // phpcs:ignore Squiz.PHP.NonExecutableCode
            yield;
        }

        yield $t;
    }

    /**
     * @param Node $node
     * @return array
     */
    private function getBasicDetails(Node $node)
    {
        $details = [];
        if (isset($node->class)) {
            $details['static'] = true;
        }
        if (isset($node->args) && count($node->args) > 0) {
            $details['arguments'] = [];
            foreach ($node->args as $arg) {
                    $details['arguments'][] =
                    [
                        'type' => $arg->value->runtimeType->__toString(),
                        'value' => $arg->value->value ?? null
                    ];
            };
        }
        if (isset($node->class) && $node->class instanceof Node\Name && (string)$node->class === 'parent') {
            $details['onParent'] = true;
        }
        return $details;
    }

    /**
     * @param array $directClasses
     * @param ClassLikeMemberDeclaration $member
     * @return array
     */
    private function usedMemberDetails(array $directClasses, ClassLikeMemberDeclaration $member): array
    {
        $details = [];
        if (!in_array($member->getClassLikeName(), $directClasses)) {
            $details['asInherited'] = true;
        }
        return $details;
    }
}
