<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\Usage;

use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeMemberDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ConstantDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\InCodeDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\MethodDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\PropertyDeclaration;
use PhpParser\Node;
use function sprintf;

class InheritedMember implements Probe
{
    /**
     * @inheritDoc
     */
    public function check(Node $node): array
    {
        if (!$node instanceof Node\Stmt\ClassLike) {
            return [];
        }

        $declaration = $node->getAttribute('declaration');
        if (!$declaration instanceof ClassLikeDeclaration\ClassLike) {
            return [];
        }

        $usages = [];
        /** @var ClassLikeMemberDeclaration $inherited */
        foreach ($declaration->getInheritedMembers() as $inheritedMemberInfo) {
            $inherited = $inheritedMemberInfo['inherited'];
            $own = $inheritedMemberInfo['own'];
            $usages[] = new UsageCase(
                sprintf(
                    '%s::%s%s',
                    $inherited->getClassLikeName(),
                    $inherited instanceof PropertyDeclaration ? '$' : '',
                    $inherited->getName()
                ),
                sprintf('%s-%s', $inheritedMemberInfo['type'], $this->getMemberType($inheritedMemberInfo['inherited'])),
                $this->getPosition($node, $own)
            );
        }
        return $usages;
    }

    /**
     * @param Node $node
     * @param ClassLikeMemberDeclaration|null $own
     * @return array
     */
    private function getPosition(Node $node, ?ClassLikeMemberDeclaration $own)
    {
        if ($own instanceof InCodeDeclaration) {
            $node = $own->getDeclarationNodes();
            $node = $node[count($node) - 1];
        }
        return [
            'startLine' => $node->getStartLine(),
            'endLine' => $node->getEndLine(),
            'startFilePos' => $node->getStartFilePos(),
            'endFilePos' => $node->getEndFilePos(),
        ];
    }

    /**
     * @param ClassLikeMemberDeclaration $d
     * @return string
     */
    private function getMemberType(ClassLikeMemberDeclaration $d)
    {
        if ($d instanceof MethodDeclaration) {
            return 'method';
        } elseif ($d instanceof PropertyDeclaration) {
            return 'property';
        } elseif ($d instanceof ConstantDeclaration) {
            return 'constant';
        } else {
            return 'unknown';
        }
    }
}
