<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\CodeStructuralElement\Php\Usage;

use PhpParser\Node;
use PhpParser\Node\Param;
use PhpParser\Node\Stmt\Class_;
use PhpParser\Node\Stmt\ClassMethod;

class MethodExtended implements Probe
{
    /**
     * @inheritDoc
     */
    public function check(Node $node): array
    {
        if (!$node instanceof ClassMethod) {
            return [];
        }

        if ($node->name->name === '__construct') {
            return [];
        }

        if ($node->isPrivate()) {
            return [];
        }

        /** @var Class_ $class */
        $class = $node->getAttribute('parent');
        if (!$class->extends) {
            return [];
        }

        $extends = is_array($class->extends) ? reset($class->extends) : $class->extends;

        return [
            new UsageCase(
                $extends . '::' . $node->name,
                'extend-method',
                [
                    'position' => [
                        'startLine' => $node->getStartLine(),
                        'endLine' => $node->getEndLine(),
                        'startFilePos' => $node->getStartFilePos(),
                        'endFilePos' => $node->getEndFilePos(),
                    ],
                ],
                [
                    'arguments' => array_map(
                        function (Param $parameter) {
                            return [
                                'name' => $parameter->var->name,
                                'type' => $this->toString($parameter->type ?? $parameter->var->runtimeType),
                                'optional' => $parameter->default !== null
                            ];
                        },
                        $node->params
                    ),
                    'returnType' => $this->toString($node->getReturnType() ?: $this->getReturnType($node))
                ]
            )
        ];
    }

    /**
     * Convert anything to string
     *
     * @param mixed $value
     * @return string
     */
    private function toString($value): string
    {
        if (is_object($value)) {
            return method_exists($value, '__toString') ? (string) $value : '';
        }
        return is_array($value) ? '' : (string) $value;
    }

    /**
     * @param Node $node
     * @return string|null
     */
    private function getReturnType(Node $node): ?string
    {
        if (!$node->hasAttribute('docComment')) {
            return null;
        }
        $annotations = $node->getAttribute('docComment')->children ?? [];

        foreach ($annotations as $tag) {
            if (isset($tag->name)
                && $tag->name === '@return'
                && is_object($tag->value)
                && property_exists($tag->value, 'type')
            ) {
                return (string) $tag->value->type;
            }
        }
        return null;
    }
}
