<?php
/**
 * Copyright 2022 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Composer;

use Exception;
use InvalidArgumentException;
use RuntimeException;

class MagentoDownloader
{
    /**
     * Information about delivery profiles can be gathered from
     * https://experienceleague.adobe.com/docs/commerce-operations/release/availability.html
     *
     * Each array key is a Magento CE/EE version number, and its value array contains all required extensions
     * alongside its versions for that release
     */
    private const RELEASE_VERSIONS = [
        '2.3.0' => [
            'magento/extension-b2b' => '1.1.0',
            'magento/module-paypal-on-boarding' => '100.3.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '100.3.0',
            'magento/module-gift-card-sample-data' => '100.3.0',
            'magento/module-gift-registry-sample-data' => '100.3.0',
            'magento/module-multiple-wishlist-sample-data' => '100.3.0',
            'magento/module-target-rule-sample-data' => '100.3.0',
            'fastly/magento2' => '^1.2',
        ],
        '2.3.1' => [
            'magento/extension-b2b' => '1.1.1',
            'magento/module-paypal-on-boarding' => '100.3.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '100.3.1',
            'magento/module-gift-card-sample-data' => '100.3.1',
            'magento/module-gift-registry-sample-data' => '100.3.1',
            'magento/module-multiple-wishlist-sample-data' => '100.3.1',
            'magento/module-target-rule-sample-data' => '100.3.1',
            'fastly/magento2' => '^1.2',
        ],
        '2.3.2' => [
            'magento/extension-b2b' => '1.1.2',
            'magento/module-paypal-on-boarding' => '100.3.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '100.3.2-p2',
            'magento/module-gift-card-sample-data' => '100.3.2-p2',
            'magento/module-gift-registry-sample-data' => '100.3.2-p2',
            'magento/module-multiple-wishlist-sample-data' => '100.3.2-p2',
            'magento/module-target-rule-sample-data' => '100.3.2-p2',
            'fastly/magento2' => '^1.2',
        ],
        '2.3.2-p2' => [
            'magento/extension-b2b' => '1.1.2-p2',
            'magento/module-paypal-on-boarding' => '100.3.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '100.3.2-p2',
            'magento/module-gift-card-sample-data' => '100.3.2-p2',
            'magento/module-gift-registry-sample-data' => '100.3.2-p2',
            'magento/module-multiple-wishlist-sample-data' => '100.3.2-p2',
            'magento/module-target-rule-sample-data' => '100.3.2-p2',
            'fastly/magento2' => '^1.2',
        ],
        '2.3.3' => [
            'magento/extension-b2b' => '1.1.3',
            'magento/module-paypal-on-boarding' => '100.3.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '100.3.3',
            'magento/module-gift-card-sample-data' => '100.3.3',
            'magento/module-gift-registry-sample-data' => '100.3.3',
            'magento/module-multiple-wishlist-sample-data' => '100.3.3',
            'magento/module-target-rule-sample-data' => '100.3.3',
            'fastly/magento2' => '^1.2',
        ],
        '2.3.3-p1' => [
            'magento/extension-b2b' => '1.1.3-p1',
            'magento/module-paypal-on-boarding' => '100.3.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '100.3.3',
            'magento/module-gift-card-sample-data' => '100.3.3',
            'magento/module-gift-registry-sample-data' => '100.3.3',
            'magento/module-multiple-wishlist-sample-data' => '100.3.3',
            'magento/module-target-rule-sample-data' => '100.3.3',
            'fastly/magento2' => '^1.2',
        ],
        '2.3.4' => [
            'magento/extension-b2b' => '1.1.4',
            'magento/module-paypal-on-boarding' => '100.3.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '100.3.4',
            'magento/module-gift-card-sample-data' => '100.3.4',
            'magento/module-gift-registry-sample-data' => '100.3.4',
            'magento/module-multiple-wishlist-sample-data' => '100.3.4',
            'magento/module-target-rule-sample-data' => '100.3.4',
            'fastly/magento2' => '^1.2',
        ],
        '2.3.4-p1' => [
            'magento/extension-b2b' => '1.1.4-p1',
            'magento/module-paypal-on-boarding' => '100.3.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '100.3.4',
            'magento/module-gift-card-sample-data' => '100.3.4',
            'magento/module-gift-registry-sample-data' => '100.3.4',
            'magento/module-multiple-wishlist-sample-data' => '100.3.4',
            'magento/module-target-rule-sample-data' => '100.3.4',
            'fastly/magento2' => '^1.2',
        ],
        '2.3.4-p2' => [
            'magento/extension-b2b' => '1.1.4-p2',
            'magento/module-paypal-on-boarding' => '100.3.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '100.3.4',
            'magento/module-gift-card-sample-data' => '100.3.4',
            'magento/module-gift-registry-sample-data' => '100.3.4',
            'magento/module-multiple-wishlist-sample-data' => '100.3.4',
            'magento/module-target-rule-sample-data' => '100.3.4',
            'fastly/magento2' => '^1.2',
        ],
        '2.3.5' => [
            'magento/extension-b2b' => '1.1.5',
            'magento/module-paypal-on-boarding' => '100.3.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '100.3.5',
            'magento/module-gift-card-sample-data' => '100.3.5',
            'magento/module-gift-registry-sample-data' => '100.3.5',
            'magento/module-multiple-wishlist-sample-data' => '100.3.5',
            'magento/module-target-rule-sample-data' => '100.3.5',
            'fastly/magento2' => '^1.2',
        ],
        '2.3.5-p1' => [
            'magento/extension-b2b' => '1.1.5-p1',
            'magento/module-paypal-on-boarding' => '100.3.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '100.3.5',
            'magento/module-gift-card-sample-data' => '100.3.5',
            'magento/module-gift-registry-sample-data' => '100.3.5',
            'magento/module-multiple-wishlist-sample-data' => '100.3.5',
            'magento/module-target-rule-sample-data' => '100.3.5',
            'fastly/magento2' => '^1.2',
        ],
        '2.3.5-p2' => [
            'magento/extension-b2b' => '1.1.5-p2',
            'magento/module-paypal-on-boarding' => '100.3.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '100.3.5',
            'magento/module-gift-card-sample-data' => '100.3.5',
            'magento/module-gift-registry-sample-data' => '100.3.5',
            'magento/module-multiple-wishlist-sample-data' => '100.3.5',
            'magento/module-target-rule-sample-data' => '100.3.5',
            'fastly/magento2' => '^1.2',
        ],
        '2.3.6' => [
            'magento/extension-b2b' => '1.1.6',
            'magento/module-paypal-on-boarding' => '100.3.1',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '100.3.5',
            'magento/module-gift-card-sample-data' => '100.3.5',
            'magento/module-gift-registry-sample-data' => '100.3.5',
            'magento/module-multiple-wishlist-sample-data' => '100.3.5',
            'magento/module-target-rule-sample-data' => '100.3.5',
            'fastly/magento2' => '^1.2',
        ],
        '2.3.6-p1' => [
            'magento/extension-b2b' => '1.1.6-p1',
            'magento/module-paypal-on-boarding' => '100.3.1',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '100.3.5',
            'magento/module-gift-card-sample-data' => '100.3.5',
            'magento/module-gift-registry-sample-data' => '100.3.5',
            'magento/module-multiple-wishlist-sample-data' => '100.3.5',
            'magento/module-target-rule-sample-data' => '100.3.5',
            'fastly/magento2' => '^1.2',
        ],
        '2.3.7' => [
            'magento/extension-b2b' => '1.1.7',
            'magento/module-paypal-on-boarding' => '100.3.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '100.3.5',
            'magento/module-gift-card-sample-data' => '100.3.5',
            'magento/module-gift-registry-sample-data' => '100.3.5',
            'magento/module-multiple-wishlist-sample-data' => '100.3.5',
            'magento/module-target-rule-sample-data' => '100.3.5',
            'fastly/magento2' => '^1.2',
        ],
        '2.3.7-p1' => [
            'magento/extension-b2b' => '1.1.7-p1',
            'magento/module-paypal-on-boarding' => '100.3.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '100.3.5',
            'magento/module-gift-card-sample-data' => '100.3.5',
            'magento/module-gift-registry-sample-data' => '100.3.5',
            'magento/module-multiple-wishlist-sample-data' => '100.3.5',
            'magento/module-target-rule-sample-data' => '100.3.5',
            'fastly/magento2' => '^1.2',
        ],
        '2.3.7-p2' => [
            'magento/extension-b2b' => '1.1.7-p2',
            'magento/module-paypal-on-boarding' => '100.3.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '100.3.5',
            'magento/module-gift-card-sample-data' => '100.3.5',
            'magento/module-gift-registry-sample-data' => '100.3.5',
            'magento/module-multiple-wishlist-sample-data' => '100.3.5',
            'magento/module-target-rule-sample-data' => '100.3.5',
            'fastly/magento2' => '^1.2',
        ],
        '2.3.7-p3' => [
            'magento/extension-b2b' => '1.1.7-p3',
            'magento/module-paypal-on-boarding' => '100.3.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '100.3.5',
            'magento/module-gift-card-sample-data' => '100.3.5',
            'magento/module-gift-registry-sample-data' => '100.3.5',
            'magento/module-multiple-wishlist-sample-data' => '100.3.5',
            'magento/module-target-rule-sample-data' => '100.3.5',
            'fastly/magento2' => '^1.2',
        ],
        '2.3.7-p4' => [
            'magento/extension-b2b' => '1.1.7-p4',
            'magento/module-paypal-on-boarding' => '100.3.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '100.3.5',
            'magento/module-gift-card-sample-data' => '100.3.5',
            'magento/module-gift-registry-sample-data' => '100.3.5',
            'magento/module-multiple-wishlist-sample-data' => '100.3.5',
            'magento/module-target-rule-sample-data' => '100.3.5',
            'fastly/magento2' => '^1.2',
        ],
        '2.4.0' => [
            'magento/extension-b2b' => '1.2.0',
            'magento/module-paypal-on-boarding' => '100.4.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '^100.4',
            'magento/module-gift-card-sample-data' => '^100.4',
            'magento/module-gift-registry-sample-data' => '^100.4',
            'magento/module-multiple-wishlist-sample-data' => '^100.4',
            'magento/module-target-rule-sample-data' => '^100.4',
            'fastly/magento2' => '^1.2',
        ],
        '2.4.0-p1' => [
            'magento/extension-b2b' => '1.2.0-p1',
            'magento/module-paypal-on-boarding' => '100.4.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '^100.4',
            'magento/module-gift-card-sample-data' => '^100.4',
            'magento/module-gift-registry-sample-data' => '^100.4',
            'magento/module-multiple-wishlist-sample-data' => '^100.4',
            'magento/module-target-rule-sample-data' => '^100.4',
            'fastly/magento2' => '^1.2',
        ],
        '2.4.1' => [
            'magento/extension-b2b' => '1.3.0',
            'magento/module-paypal-on-boarding' => '100.4.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '^100.4',
            'magento/module-gift-card-sample-data' => '^100.4',
            'magento/module-gift-registry-sample-data' => '^100.4',
            'magento/module-multiple-wishlist-sample-data' => '^100.4',
            'magento/module-target-rule-sample-data' => '^100.4',
            'fastly/magento2' => '^1.2',
        ],
        '2.4.1-p1' => [
            'magento/extension-b2b' => '1.3.0-p1',
            'magento/module-paypal-on-boarding' => '100.4.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '^100.4',
            'magento/module-gift-card-sample-data' => '^100.4',
            'magento/module-gift-registry-sample-data' => '^100.4',
            'magento/module-multiple-wishlist-sample-data' => '^100.4',
            'magento/module-target-rule-sample-data' => '^100.4',
            'fastly/magento2' => '^1.2',
        ],
        '2.4.2' => [
            'magento/extension-b2b' => '1.3.1',
            'magento/module-paypal-on-boarding' => '100.4.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '^100.4',
            'magento/module-gift-card-sample-data' => '^100.4',
            'magento/module-gift-registry-sample-data' => '^100.4',
            'magento/module-multiple-wishlist-sample-data' => '^100.4',
            'magento/module-target-rule-sample-data' => '^100.4',
            'fastly/magento2' => '^1.2',
        ],
        '2.4.2-p1' => [
            'magento/extension-b2b' => '1.3.1-p1',
            'magento/module-paypal-on-boarding' => '100.4.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '^100.4',
            'magento/module-gift-card-sample-data' => '^100.4',
            'magento/module-gift-registry-sample-data' => '^100.4',
            'magento/module-multiple-wishlist-sample-data' => '^100.4',
            'magento/module-target-rule-sample-data' => '^100.4',
            'fastly/magento2' => '^1.2',
        ],
        '2.4.2-p2' => [
            'magento/extension-b2b' => '1.3.1-p2',
            'magento/module-paypal-on-boarding' => '100.4.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '^100.4',
            'magento/module-gift-card-sample-data' => '^100.4',
            'magento/module-gift-registry-sample-data' => '^100.4',
            'magento/module-multiple-wishlist-sample-data' => '^100.4',
            'magento/module-target-rule-sample-data' => '^100.4',
            'fastly/magento2' => '^1.2',
        ],
        '2.4.3' => [
            'magento/extension-b2b' => '1.3.2',
            'magento/module-paypal-on-boarding' => '100.4.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '^100.4',
            'magento/module-gift-card-sample-data' => '^100.4',
            'magento/module-gift-registry-sample-data' => '^100.4',
            'magento/module-multiple-wishlist-sample-data' => '^100.4',
            'magento/module-target-rule-sample-data' => '^100.4',
            'fastly/magento2' => '^1.2',
        ],
        '2.4.3-p1' => [
            'magento/extension-b2b' => '1.3.2-p1',
            'magento/module-paypal-on-boarding' => '100.4.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '^100.4',
            'magento/module-gift-card-sample-data' => '^100.4',
            'magento/module-gift-registry-sample-data' => '^100.4',
            'magento/module-multiple-wishlist-sample-data' => '^100.4',
            'magento/module-target-rule-sample-data' => '^100.4',
            'fastly/magento2' => '^1.2',
        ],
        '2.4.3-p2' => [
            'magento/extension-b2b' => '1.3.2-p2',
            'magento/module-paypal-on-boarding' => '100.4.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '^100.4',
            'magento/module-gift-card-sample-data' => '^100.4',
            'magento/module-gift-registry-sample-data' => '^100.4',
            'magento/module-multiple-wishlist-sample-data' => '^100.4',
            'magento/module-target-rule-sample-data' => '^100.4',
            'fastly/magento2' => '^1.2',
        ],
        '2.4.3-p3' => [
            'magento/extension-b2b' => '1.3.2-p3',
            'magento/module-paypal-on-boarding' => '100.4.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '^100.4',
            'magento/module-gift-card-sample-data' => '^100.4',
            'magento/module-gift-registry-sample-data' => '^100.4',
            'magento/module-multiple-wishlist-sample-data' => '^100.4',
            'magento/module-target-rule-sample-data' => '^100.4',
            'fastly/magento2' => '^1.2',
        ],
        '2.4.4' => [
            'magento/extension-b2b' => '1.3.3',
            'magento/module-paypal-on-boarding' => '100.4.2',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '^100.4',
            'magento/module-gift-card-sample-data' => '^100.4',
            'magento/module-gift-registry-sample-data' => '^100.4',
            'magento/module-multiple-wishlist-sample-data' => '^100.4',
            'magento/module-target-rule-sample-data' => '^100.4',
            'fastly/magento2' => '^1.2',
        ],
        '2.4.4-p1' => [
            'magento/extension-b2b' => '1.3.3-p1',
            'magento/module-paypal-on-boarding' => '~100.4.0',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '^100.4',
            'magento/module-gift-card-sample-data' => '^100.4',
            'magento/module-gift-registry-sample-data' => '^100.4',
            'magento/module-multiple-wishlist-sample-data' => '^100.4',
            'magento/module-target-rule-sample-data' => '^100.4',
            'fastly/magento2' => '^1.2.34',
        ],
        '2.4.4-p2' => [
            'magento/extension-b2b' => '1.3.3-p2',
            'magento/module-paypal-on-boarding' => '~100.4.0',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '^100.4',
            'magento/module-gift-card-sample-data' => '^100.4',
            'magento/module-gift-registry-sample-data' => '^100.4',
            'magento/module-multiple-wishlist-sample-data' => '^100.4',
            'magento/module-target-rule-sample-data' => '^100.4',
            'fastly/magento2' => '^1.2.34',
        ],
        '2.4.5' => [
            'magento/extension-b2b' => '1.3.4',
            'magento/module-paypal-on-boarding' => '~100.4.0',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '^100.4',
            'magento/module-gift-card-sample-data' => '^100.4',
            'magento/module-gift-registry-sample-data' => '^100.4',
            'magento/module-multiple-wishlist-sample-data' => '^100.4',
            'magento/module-target-rule-sample-data' => '^100.4',
            'fastly/magento2' => '^1.2.34',
        ],
        '2.4.5-p1' => [
            'magento/extension-b2b' => '1.3.4-p1',
            'magento/module-paypal-on-boarding' => '~100.4.0',
            'magento/magento-cloud-components' => '1.0.10',
            'magento/module-customer-balance-sample-data' => '^100.4',
            'magento/module-gift-card-sample-data' => '^100.4',
            'magento/module-gift-registry-sample-data' => '^100.4',
            'magento/module-multiple-wishlist-sample-data' => '^100.4',
            'magento/module-target-rule-sample-data' => '^100.4',
            'fastly/magento2' => '^1.2.34',
        ],
    ];

    /**
     * @param  string $dir
     * @param  string $version
     * @throws Exception
     */
    public function execute(string $dir, string $version)
    {
        $toDirectory = $this->readTargetDir($dir);
        $packagePath = $toDirectory . '/' . $version;
        if (is_dir($packagePath)) {
            throw new Exception(
                sprintf(
                    'Installing Adobe Commerce (%s): Already available',
                    $version
                )
            );
        }
        $this->executeCreateProject($version, $packagePath);
        $this->executeRequire($version, $packagePath);
    }

    /**
     * @param string $version
     * @param string $packagePath
     */
    private function executeRequire(string $version, string $packagePath)
    {
        $packages = [];
        if (!array_key_exists($version, self::RELEASE_VERSIONS)) {
            throw new RuntimeException(sprintf("Version %s not found in released versions", $version));
        }

        foreach (self::RELEASE_VERSIONS[$version] as $currentPackage => $packageVersion) {
            $packages[] = $currentPackage.":".$packageVersion;
        }
        $packages = implode(' ', $packages);

        $this->exec(
            sprintf(
                "composer require magento/project-enterprise-edition:%s %s ".
                "--ignore-platform-reqs --no-scripts --with-all-dependencies -d %s",
                $version,
                $packages,
                $packagePath
            )
        );
    }

    /**
     * @param  string $version
     * @param  string $packagePath
     * @return void
     */
    private function executeCreateProject(string $version, string $packagePath)
    {
        $this->exec(
            sprintf(
                "composer create-project --repository-url=https://repo.magento.com/ ".
                "magento/project-community-edition:%s --ignore-platform-reqs --no-scripts %s",
                $version,
                $packagePath
            )
        );
    }

    /**
     * @param  string $command
     * @return void
     */
    private function exec(string $command): void
    {
        // phpcs:ignore Magento2.Security.InsecureFunction
        exec($command, $output, $resultCode);

        if ($resultCode !== null && $resultCode !== 0) {
            throw new RuntimeException(sprintf("Exec call '%s' failed", $command));
        }
    }

    /**
     * @param  string $dir
     * @return string
     */
    private function readTargetDir(string $dir): string
    {
        if (!file_exists($dir)) {
            mkdir($dir, 0755, true);
            clearstatcache(true, $dir);
        }
        if (!is_dir($dir)) {
            throw new InvalidArgumentException("Output path '$dir' exists and is not a directory.");
        }
        return realpath($dir);
    }
}
