<?php
/**
 * Copyright 2022 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Composer;

use Composer\Composer;
use Composer\Config;
use Composer\Downloader\DownloaderInterface;
use Composer\EventDispatcher\EventDispatcher;
use Composer\Factory;
use Composer\Filter\PlatformRequirementFilter\IgnoreAllPlatformRequirementFilter;
use Composer\Installer;
use Composer\IO\ConsoleIO;
use Composer\IO\IOInterface;
use Composer\Package\BasePackage;
use Composer\Package\Package;
use Composer\Package\PackageInterface;
use Composer\Package\RootPackage;
use Composer\Repository\RepositoryFactory;
use Composer\Repository\RepositoryManager;
use Composer\Util\ErrorHandler;
use Composer\Util\HttpDownloader;
use Composer\Util\Loop;
use Composer\Util\ProcessExecutor;
use Exception;
use FilesystemIterator;
use React\Promise\PromiseInterface;
use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;

class PackagesDownloader
{
    /**
     * @var ConsoleIO
     */
    private $io;

    /**
     * @var Config
     */
    private $config;

    /**
     * @var Composer
     */
    private $composer;

    private const VERSION = '1';

    /**
     * @param Config    $config
     * @param ConsoleIO $io
     * @param array     $auth
     */
    public function __construct(Config $config, ConsoleIO $io, array $auth)
    {
        $this->io = $io;
        ErrorHandler::register($io);

        $this->config = $config;

        $this->config->merge(
            [
            'repositories' => [
                'packagist.org' => false,
            ],
            'config' => $auth,
            'cache-files-ttl' => 0,
            ]
        );

        $this->composer = new Composer();
        $this->composer->setPackage(new RootPackage('magento', self::VERSION, self::VERSION));
    }

    /**
     * @param string $dir
     * @param array $sources
     * @param array $require
     * @param string $stability
     * @param bool $install
     * @param bool $showDiskUsage
     * @param bool $allVersions
     */
    public function execute(
        string $dir,
        array  $sources,
        array  $require,
        string $stability,
        bool   $install,
        bool   $showDiskUsage,
        bool   $allVersions
    ) {
        foreach ($sources as $source) {
            $this->config->merge(
                [
                'repositories' => [
                    $source => [
                        'type' => 'composer',
                        'url' => $source,
                    ]
                ]
                ]
            );
        }
        $httpDownloader = Factory::createHttpDownloader($this->io, $this->config);
        $eventDispatcher = new EventDispatcher($this->composer, $this->io);
        $eventDispatcher->setRunScripts(false);
        $processExecutor = new ProcessExecutor($this->io);

        $repositoryManager = $this->createRepositoryManager(
            $this->io,
            $this->config,
            $httpDownloader,
            $eventDispatcher
        );
        $this->composer->setDownloadManager(
            (new Factory())->createDownloadManager(
                $this->io,
                $this->config,
                $httpDownloader,
                $processExecutor,
                $eventDispatcher
            )
        );
        $this->composer->setLoop(new Loop($httpDownloader, $processExecutor));

        $packages = (new PackagesInformation())->execute($repositoryManager, $require, $stability, $allVersions);
        if (empty($packages)) {
            $this->io->warning('Required packages not found');
        }
        $totalCountToProcess = count($packages);
        $processedCount = 0;
        $dataSize = 0;

        $packageDownloader = new SinglePackageDownloader();
        /**
         * @var $package BasePackage
        */
        foreach ($packages as $package) {
            $packagePath = sprintf(
                '%s/%s/%s',
                $dir,
                $package->getPrettyName(),
                $package->getPrettyVersion()
            );
            if (is_dir($packagePath)) {
                $this->io->warning(
                    sprintf(
                        '  - Installing %s (%s): Already available',
                        trim($package->getPrettyName()),
                        $package->getPrettyVersion()
                    )
                );
            }
            try {
                $packageDownloader->execute($this->composer, $package, $packagePath, $install, $this->io);
                $processedCount++;
                if ($showDiskUsage) {
                    $packageSize = $this->dirSize($packagePath);
                    $dataSize += $packageSize;
                    $this->io->write(
                        sprintf(
                            '    Package size: %s, Disk usage: %s, Processed: %s/%s (%d%%)',
                            $this->humanBytes($packageSize),
                            $this->humanBytes($dataSize),
                            $processedCount,
                            $totalCountToProcess,
                            $processedCount * 100 / $totalCountToProcess
                        )
                    );
                } else {
                    $this->io->write(
                        sprintf(
                            '    Processed: %s/%s (%d%%)',
                            $processedCount,
                            $totalCountToProcess,
                            $processedCount * 100 / $totalCountToProcess
                        )
                    );
                }
            } catch (\Exception $e) {
                $this->io->warning("Package $package failed to download");
            }
        }

        if ($showDiskUsage) {
            $this->io->write(sprintf('Done! Disk usage: %s.', $this->humanBytes($dataSize)));
        } else {
            $this->io->write('Done!');
        }
    }

    /**
     * @param  IOInterface     $io
     * @param  Config          $config
     * @param  HttpDownloader  $httpDownloader
     * @param  EventDispatcher $eventDispatcher
     * @return RepositoryManager
     */
    private function createRepositoryManager(
        IOInterface     $io,
        Config          $config,
        HttpDownloader  $httpDownloader,
        EventDispatcher $eventDispatcher
    ): RepositoryManager {
        $repositoryManager = RepositoryFactory::manager($io, $config, $httpDownloader, $eventDispatcher);

        foreach (RepositoryFactory::defaultRepos($io, $config, $repositoryManager) as $repository) {
            $repositoryManager->addRepository($repository);
        };
        return $repositoryManager;
    }

    /**
     * @param  int $bytes
     * @return string
     */
    private function humanBytes(int $bytes): string
    {
        if (!$bytes) {
            return '0';
        }

        static $sizes = ['B', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB'];
        $i = floor(log($bytes) / log(1024));
        return sprintf('%.02F', $bytes / pow(1024, $i)) * 1 . ' ' . $sizes[$i];
    }

    /**
     * @param  string $dir
     * @return int
     */
    private function dirSize(string $dir): int
    {
        $size = 0;
        foreach (new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dir, FilesystemIterator::SKIP_DOTS)
        ) as $f) {
            $size += $f->getSize();
        }
        return $size;
    }
}
