<?php
/**
 * Copyright 2022 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
namespace Magento\Mray\Composer;

use Composer\Package\PackageInterface;
use Composer\Repository\RepositoryManager;

class PackagesInformation
{
    private const STABILITY = [
        'dev' => 1,
        'alpha' => 2,
        'beta' => 3,
        'rc' => 4,
        'stable' => 5,
    ];

    /**
     * @param RepositoryManager $repositoryManager
     * @param array $requiredPackages
     * @param string $stability
     * @param bool $allVersions
     * @return array PackageInterface[]|null
     */
    public function execute(
        RepositoryManager $repositoryManager,
        array $requiredPackages,
        string $stability,
        bool $allVersions = false
    ): array {
        $repos = $repositoryManager->getRepositories();
        if (empty($requiredPackages)) {
            return $this->getAllPackages($repos, $stability, $allVersions);
        }
        return $this->getRequiredPackages($repos, $requiredPackages, $stability, $allVersions);
    }

    /**
     * @param array $repos
     * @param string $stability
     * @param bool $allVersions
     * @return array
     */
    private function getAllPackages(array $repos, string $stability, bool $allVersions): array
    {
        $packages = [];
        foreach ($repos as $repo) {
            $names = $repo->getPackageNames();
            foreach ($names as $name) {
                $newPackages = $repo->findPackages($name);
                $packages = $this->getPackagesMatchingStability($newPackages, $stability, $packages, $allVersions);
            }
        }
        return $packages;
    }

    /**
     * @param array $repos
     * @param array $requiredPackages
     * @param string $stability
     * @param bool $allVersions
     * @return array
     */
    private function getRequiredPackages(
        array $repos,
        array $requiredPackages,
        string $stability,
        bool $allVersions
    ): array {
        $packages = [];
        foreach ($repos as $repo) {
            foreach ($requiredPackages as $require) {
                $packageName = $require;
                $packageVersion = null;
                if (strpos($require, ':') !== false) {
                    list($packageName, $packageVersion) = explode(':', $require);
                }
                $newPackages = $repo->findPackages($packageName, $packageVersion);
                $packages = $this->getPackagesMatchingStability($newPackages, $stability, $packages, $allVersions);
            }
        }
        return $packages;
    }

    /**
     * @param array $newPackages
     * @param string $stability
     * @param array $packages
     * @param bool $allVersions
     * @return array
     */
    private function getPackagesMatchingStability(
        array $newPackages,
        string $stability,
        array $packages,
        bool $allVersions
    ): array {
        foreach ($newPackages as $new) {
            if ($this->hasRequiredMinimalStability($new, $stability)) {
                $packages[] = $new;
                if (!$allVersions) {
                    break;
                }
            }
        }
        return $packages;
    }

    /**
     * @param  PackageInterface $package
     * @param  string           $requiredStability
     * @return bool
     */
    private function hasRequiredMinimalStability(PackageInterface $package, string $requiredStability): bool
    {
        return self::STABILITY[$requiredStability] <= self::STABILITY[strtolower($package->getStability())];
    }
}
