<?php
/**
 * Copyright 2022 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Console\Command;

use Composer\Factory;
use Composer\IO\ConsoleIO;
use Composer\Downloader\TransportException;
use Exception;
use JsonException;
use Magento\Mray\Composer\PackagesDownloader;
use RuntimeException;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Helper\HelperSet;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class DownloadRepoPackages extends Command
{
    /**
     * @var string
     */
    protected static $defaultName = 'crawler:download-repo-packages';

    /**
     * @inheritDoc
     */
    protected function configure()
    {
        $this->addArgument(
            'out',
            InputArgument::REQUIRED,
            'The directory where to fetch packages.'
        );
        $this->addOption(
            'source',
            's',
            InputOption::VALUE_REQUIRED | InputOption::VALUE_IS_ARRAY,
            'Source configuration. May be a Composer repository URL, path to composer.json file or JSON.',
            [
                'https://repo.magento.com/',
            ]
        );
        $this->addOption(
            'require',
            'r',
            InputOption::VALUE_REQUIRED | InputOption::VALUE_IS_ARRAY,
            'Fetch only packages that match constraint. If not specified all packages fetched.',
        );
        $this->addOption(
            'minimum-stability',
            'm',
            InputOption::VALUE_REQUIRED,
            'Required stability for packages to be downloaded.',
            'stable'
        );
        $this->addOption(
            'install',
            'i',
            InputOption::VALUE_NONE,
            'Install dependencies for fetched packages.',
        );
        $this->addOption(
            'show-disk-usage',
            'd',
            InputOption::VALUE_NONE,
            'Whether to show disk usage or not.'
        );
        $this->addOption(
            'auth',
            'a',
            InputOption::VALUE_REQUIRED,
            'Auth configuration. COMPOSER_AUTH environment variable may be used instead.',
        );
        $this->addOption(
            'all-versions',
            'l',
            InputOption::VALUE_NONE,
            'Install all versions of fetched packages. Only latest one will be downloaded if omitted.',
        );
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     * @return int
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        try {
            $dir = $input->getArgument('out');
            $source = $input->getOption('source');
            $require = $input->getOption('require');
            $stability = $input->getOption('minimum-stability');
            $install = $input->getOption('install');
            $showDiskUsage = $input->getOption('show-disk-usage');
            $allVersions = $input->getOption('all-versions');
            $auth = $this->readAuthConfig($input);

            $io = new ConsoleIO($input, $output, new HelperSet());
            $downloader = new PackagesDownloader(Factory::createConfig($io, $dir), $io, $auth ?? []);
            $downloader->execute($dir, $source, $require, $stability, $install, $showDiskUsage, $allVersions);
        } catch (TransportException $e) {
            // downloading should continue despite of Curl CURLE_URL_MALFORMAT error for current package
            $output->writeln("<error>[Composer] " . $e->getMessage() . "</error>");
            return 0;
        } catch (Exception $e) {
            $output->writeln("<error>[mray] " . $e->getMessage() . "</error>");
            return 1;
        }
        return 0;
    }

    /**
     * @param InputInterface $input
     * @return array|null
     */
    private function readAuthConfig(InputInterface $input): ?array
    {
        $json = $input->getOption('auth');
        if (!$json) {
            return null;
        }

        if (file_exists($json)) {
            $json = file_get_contents($json);
        }

        try {
            return json_decode($json, true, 10, JSON_THROW_ON_ERROR);
        } catch (JsonException $e) {
            throw new RuntimeException(sprintf(
                'Unable to read auth configuration: %s.',
                $e->getMessage(),
            ), $e->getCode(), $e);
        }
    }
}
