<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Collection;

use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassConstantDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeMemberDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\FunctionDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\GlobalConstantDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\InCodeDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\InterfaceDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\MethodDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\PropertyDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\StructuralElementDeclaration;
use Magento\Mray\Index\Data\DataTransferObject;
use Magento\Mray\Index\PhpMethodMeta;
use Magento\Mray\Index\PhpPropertyMeta;
use Magento\Mray\Index\PhpClassMeta;
use Magento\Mray\Index\PhpInterfaceMeta;
use Magento\Mray\Index\PhpClassConstantMeta;
use Magento\Mray\Index\Data\PhpFunctionMeta;
use Magento\Mray\Index\Data\PhpConstantMeta;
use Magento\Mray\Index\Data\PhpFqsenMeta;
use function explode;
use function spl_object_id;

class PhpCode implements DataCollection
{
    /** @var Dictionary  */
    private $dictionary;
    /** @var array  */
    private $data = [];

    /**
     * @param Dictionary|null $dictionary
     */
    public function __construct(?Dictionary $dictionary = null)
    {
        $this->dictionary = $dictionary ?? new Dictionary();
    }

    /**
     * @param string $fqsen
     * @param StructuralElementDeclaration $declaration
     */
    public function registerImplementation(string $fqsen, StructuralElementDeclaration $declaration): void
    {
        // Filter out code not marked as API
        if (!$this->isApiClassLike($declaration)) {
            return;
        }

        // postpone declaration transformation to DTO as in-code declarations have more data
        $this->data[$this->fqsenRef($fqsen)][] = $declaration;
    }

    /**
     * @param StructuralElementDeclaration $d
     * @return bool
     */
    private function isApiClassLike(StructuralElementDeclaration $d): bool
    {
        if (!$d instanceof ClassLikeDeclaration) {
            return false;
        }

        if (!$d instanceof InCodeDeclaration) {
            return false;
        }
        $phpDoc = $d->getPhpDoc();
        if ($phpDoc && $phpDoc->getTagsByName('@api')) {
            return true;
        }

        // todo: enable comment with API methods only when limit of package on the Repo increased
        return false;

//        foreach ($d->getMembers() as $m) {
//            if (!$m instanceof InCodeDeclaration) {
//                continue;
//            }
//            $phpDoc = $m->getPhpDoc();
//            if ($phpDoc && $phpDoc->getTagsByName('@api')) {
//                return true;
//            }
//        }
//
//        return false;
    }

    /**
     * @inheritDoc
     */
    public function exportData(): array
    {
        $implRefsCache = [];
        $implementations = [];
        foreach ($this->data as $fqsenRef => $impls) {
            foreach ($impls as $impl) {
                $implOID = spl_object_id($impl);
                $implRef = $implRefsCache[$implOID] ?? null;
                if ($implRef === null) {
                    $dto = $this->declarationDto($impl);
                    if (!$dto) {
                        $implRef = false;
                    } else {
                        $implRef = $this->dictionary->write($dto->pack($this->dictionary));
                    }
                    $implRefsCache[$implOID] = $implRef;
                }

                if ($implRef === false) {
                    continue;
                }

                $implementations[$fqsenRef][] = $implRef;
            }
        }
        return $implementations;
    }

    /**
     * @param array $data
     */
    public function importData(array $data): void
    {
        $this->data = $data;
    }

    /**
     * @param string $fqsen
     * @return string
     */
    private function fqsenRef(string $fqsen): string
    {
        return $this->dictionary->write(
            (new PhpFqsenMeta($fqsen))->pack($this->dictionary)
        );
    }

    /**
     * @param StructuralElementDeclaration $declaration
     * @return DataTransferObject|null
     */
    private function declarationDto(StructuralElementDeclaration $declaration): ?DataTransferObject
    {
        if ($declaration instanceof DataTransferObject) {
            return $declaration;
        } elseif ($declaration instanceof MethodDeclaration) {
            return PhpMethodMeta::createFromObject($declaration);
        } elseif ($declaration instanceof PropertyDeclaration) {
            return PhpPropertyMeta::createFromObject($declaration);
        } elseif ($declaration instanceof ClassDeclaration) {
            return PhpClassMeta::createFromObject($declaration);
        } elseif ($declaration instanceof InterfaceDeclaration) {
            return PhpInterfaceMeta::createFromObject($declaration);
        } elseif ($declaration instanceof ClassConstantDeclaration) {
            return PhpClassConstantMeta::createFromObject($declaration);
        } elseif ($declaration instanceof FunctionDeclaration) {
            return PhpFunctionMeta::createFromObject($declaration);
        } elseif ($declaration instanceof GlobalConstantDeclaration) {
            return PhpConstantMeta::createFromObject($declaration);
        } else {
            // todo: add traits support
            return null;
        }
    }
}
