<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Data;

use Magento\Mray\Index\Collection\Dictionary;

class MagentoComponentMeta extends DataTransferObject
{
    public const T_MODULE = 2;
    public const T_LIBRARY = 4;
    public const T_THEME = 8;
    public const T_LANGUAGE = 16;
    public const T_SETUP = 32;
    public const T_COMPONENT = 64;

    /** @var string  */
    protected $identifier;
    /** @var int  */
    protected $type;
    /** @var string|null  */
    protected $distributionPackage;
    /** @var string|null  */
    protected $version;
    /** @var string|null  */
    protected $path;
    /** @var string[] */
    protected $namespaces;
    /** @var string|null  */
    protected $checksum;

    /**
     * @param Dictionary $dict
     * @return array
     */
    protected function packData(Dictionary $dict): array
    {
        $packed = [];
        $packed[0] = $dict->write($this->identifier);
        $packed[1] = $this->type;
        if (isset($this->distributionPackage)) {
            $packed[2] = $dict->write($this->distributionPackage);
        }

        $packed[3] = $this->path;
        $packed[4] = $this->checksum;

        $packed[5] = $this->namespaces ?: null;

        return [
            0 => $dict->write(self::stringify($packed)),
            1 => $this->version,
        ];
    }

    /**
     * @param array $packed
     * @param Dictionary $dict
     * @return array
     */
    protected static function unpackData(array $packed, Dictionary $dict): array
    {
        $data = [];
        $data['version'] = $packed[1] ?? null;
        if (isset($packed[0])) {
            $packed = self::unstringify($dict->read($packed[0]));
            if (isset($packed[0])) {
                $data['identifier'] = $dict->read($packed[0]);
            }
            $data['type'] = $packed[1] ?? null;
            if (isset($packed[2])) {
                $data['distributionPackage'] = $dict->read($packed[2]);
            }

            $data['path'] = $packed[3] ?? null;
            $data['checksum'] = $packed[4] ?? null;
            $data['namespaces'] = $packed[5] ?? [];
        }

        return $data;
    }

    /**
     * @param mixed $data
     */
    public function __construct($data)
    {
        // todo: implement correct type error handling
        $this->identifier = $data['identifier'] ?? null;
        $this->type = $data['type'] ?? null;
        $this->distributionPackage = $data['distributionPackage'] ?? null;
        $this->version = $data['version'] ?? null;
        $this->namespaces = $data['namespaces'] ?? [];
        $this->path = $data['path'] ?? null;
        $this->checksum = $data['checksum'] ?? null;
    }
}
