<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Data;

use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeMemberDeclaration;
use Magento\Mray\Index\Collection\Dictionary;
use Magento\Mray\Index\PhpElementFlags;

abstract class PhpClassLikeMemberMeta extends DataTransferObject implements ClassLikeMemberDeclaration
{
    /**
     * @param ClassLikeMemberDeclaration $d
     * @return array
     */
    final protected static function membershipDataFromObject(ClassLikeMemberDeclaration $d)
    {
        $flags = 0;
        if ($d->isPrivate()) {
            $flags |= PhpElementFlags::PRIVATE;
        }
        if ($d->isProtected()) {
            $flags |= PhpElementFlags::PROTECTED;
        }
        if ($d->isPublic()) {
            $flags |= PhpElementFlags::PUBLIC;
        }
        return [
            'name' => $d->getName(),
            'owner' => $d->getClassLikeName(),
            'flags' => $flags,
            'checksum' => $d->getImplementationChecksum(),
        ];
    }

    /**
     * @return int
     */
    abstract public static function getKind(): int;

    /** @var string  */
    protected $name;
    /** @var string  */
    protected $owner;
    /** @var int  */
    protected $flags;
    /** @var string|null  */
    protected $checksum;

    /**
     * @param array $packed
     * @param Dictionary $dict
     * @return array
     */
    protected static function unpackData(array $packed, Dictionary $dict): array
    {
        return [
            'name' => $packed[0][1] ?? null,
            'owner' => isset($packed[1]) ? $dict->read($packed[1]) : null,
            'flags' => $packed[0][2] ?? null,
            'checksum' => $packed[2] ?? null,
        ];
    }

    /**
     * @param Dictionary $dict
     * @return array
     */
    protected function packData(Dictionary $dict): array
    {
        return [
            0 => [ // contract
                0 => static::getKind(),
                1 => $this->name,
                2 => $this->flags
            ],
            1 => $dict->write($this->owner),
            2 => $this->checksum,
        ];
    }

    /**
     * @param mixed $data
     */
    public function __construct($data)
    {
        $this->name = $data['name'] ?? null;
        $this->owner = $data['owner'] ?? null;
        $this->flags = $data['flags'] ?? 0;
        $this->checksum = $data['checksum'] ?? null;
    }

    /**
     * @return string|null
     */
    public function getName(): ?string
    {
        return $this->name;
    }

    /**
     * @return string|null
     */
    public function getClassLikeName(): ?string
    {
        return $this->owner;
    }

    /**
     * @return bool
     */
    public function isPublic(): bool
    {
        return (bool)($this->flags & PhpElementFlags::PUBLIC);
    }

    /**
     * @return bool
     */
    public function isProtected(): bool
    {
        return (bool)($this->flags & PhpElementFlags::PROTECTED);
    }

    /**
     * @return bool
     */
    public function isPrivate(): bool
    {
        return (bool)($this->flags & PhpElementFlags::PRIVATE);
    }

    /**
     * @return string|null
     */
    public function getImplementationChecksum(): ?string
    {
        return $this->checksum;
    }
}
