<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Data;

use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassConstantDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\MethodDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\PropertyDeclaration;
use Magento\Mray\Index\Collection\Dictionary;
use Magento\Mray\Index\PhpMethodMeta;
use Magento\Mray\Index\PhpPropertyMeta;
use Magento\Mray\Index\PhpClassConstantMeta;
use function array_map;
use function array_values;
use function base64_encode;
use function hash;
use function ksort;
use function str_replace;
use const SORT_FLAG_CASE;
use const SORT_NATURAL;

abstract class PhpClassLikeMeta extends DataTransferObject implements ClassLikeDeclaration
{
    /**
     * @param ClassLikeDeclaration $d
     * @return array
     */
    final protected static function dataFromObject(ClassLikeDeclaration $d)
    {
        $data = [
            'name' => $d->getName(),
            'checksum' => $d->getImplementationChecksum(),
        ];

        foreach ($d->getMembers() as $m) {
            if ($m->isPrivate()) {
                continue;
            }

            if ($m instanceof DataTransferObject) {
                $data['members'][] = $m;
            } elseif ($m instanceof MethodDeclaration) {
                $data['members'][] = PhpMethodMeta::createFromObject($m);
            } elseif ($m instanceof PropertyDeclaration) {
                $data['members'][] = PhpPropertyMeta::createFromObject($m);
            } elseif ($m instanceof ClassConstantDeclaration) {
                $data['members'][] = PhpClassConstantMeta::createFromObject($m);
            }
        }

        return $data;
    }

    /**
     * @return int
     */
    abstract public static function getKind(): int;

    /**
     * @param array $packed
     * @param Dictionary $dict
     * @return array
     */
    protected static function unpackData(array $packed, Dictionary $dict): array
    {
        $members = [];
        if (isset($packed[2])) {
            foreach ($packed[2] as $ms) {
                foreach ($ms as $mt => $mRef) {
                    $packedM = $dict->read($mRef);
                    switch ($mt) {
                        case 0:
                            $members[] = PhpMethodMeta::unpack($packedM, $dict);
                            break;
                        case 1:
                            $members[] = PhpPropertyMeta::unpack($packedM, $dict);
                            break;
                        case 2:
                            $members[] = PhpClassConstantMeta::unpack($packedM, $dict);
                            break;
                    }
                }
            }
        }
        return [
            'name' => isset($packed[0][1]) ? $dict->read($packed[0][1]) : null,
            'members' => $members ?: null,
            'checksum' => $packed[1] ?? null,
        ];
    }

    /**
     * @param Dictionary $dict
     * @return array
     */
    protected function packData(Dictionary $dict): array
    {
        $data = [
            0 => [
                0 => static::getKind(),
                1 => $dict->write($this->name),
            ],
            1 => $this->checksum
        ];
        if ($this->members) {
            foreach ($this->members as $m) {
                if ($m instanceof MethodDeclaration) {
                    $mt = 0;
                } elseif ($m instanceof PropertyDeclaration) {
                    $mt = 1;
                } elseif ($m instanceof ClassConstantDeclaration) {
                    $mt = 2;
                } else {
                    continue;
                }

                $packedM = $m->pack($dict);
                $rawMData = self::unstringify($packedM);

            // write member contract to class like contract section
                $data[0][2][$mt][$m->getName()] = $rawMData[0];
            // write full member information outside from signature
                $data[2][$mt][$m->getName()] = $dict->write($packedM);
            }
        }

        for ($i = 0; $i <= 2; $i++) {
            if (isset($data[0][2][$i])) {
                // we don't need exact signature just checksum too verify if it was changed
                // whole signature significantly increase index size
                ksort($data[0][2][$i], SORT_NATURAL | SORT_FLAG_CASE);
                $data[0][2][$i] = str_replace(
                    ['+', '/', '='],
                    ['-', '_', ''],
                    base64_encode(hash('md5', join('|', $data[0][2][$i]), true))
                );
            }
            if (isset($data[2][$i])) {
                ksort($data[2][$i], SORT_NATURAL | SORT_FLAG_CASE);
                $data[2][$i] = array_values($data[2][$i]);
            }
        }

        return $data;
    }

    /** @var string  */
    protected $name;
    /** @var PhpClassLikeMemberMeta[]|null  */
    protected $members;
    /** @var string|null  */
    protected $checksum;

    /**
     * @param mixed $data
     */
    public function __construct($data)
    {
        $this->name = $data['name'] ?? null;
        $this->members = isset($data['members']) ? array_map(function (PhpClassLikeMemberMeta $m) {
            return $m;
        }, $data['members']) : null;
        $this->checksum = $data['checksum'] ?? null;
    }

    /**
     * @return string|null
     */
    public function getName(): ?string
    {
        return $this->name;
    }

    /**
     * @inheritDoc
     */
    public function getMembers(): array
    {
        return $this->members ?? [];
    }

    /**
     * @return string|null
     */
    public function getImplementationChecksum(): ?string
    {
        return $this->checksum;
    }
}
