<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Data;

use Magento\Mray\CodeStructuralElement\Php\Reflection\FunctionDeclaration;
use Magento\Mray\Index\Collection\Dictionary;
use Magento\Mray\Index\PhpElementKinds;
use phpDocumentor\Reflection\Type;
use function array_map;
use function is_array;

class PhpFunctionMeta extends DataTransferObject implements FunctionDeclaration
{
    /**
     * @param FunctionDeclaration $d
     * @return static
     */
    public static function createFromObject(FunctionDeclaration $d): self
    {
        $data = [];
        $data['name'] = $d->getName();
        $data['returns'] = PhpTypeMeta::createFromObject($d->getReturnType());
        foreach ($d->getParams() as $param) {
            $data['params'][] = PhpParameterMeta::createFromObject($param);
        }
        $data['checksum'] = $d->getImplementationChecksum();

        return new static($data);
    }

    /**
     * @return int
     */
    public static function getKind(): int
    {
        return PhpElementKinds::DECLARED_ | PhpElementKinds::CALLABLE_;
    }

    /** @var string */
    protected $name;
    /** @var PhpTypeMeta  */
    protected $returns;
    /** @var PhpParameterMeta[]|null */
    private $params;
    /** @var string|null  */
    private $checksum;

    /**
     * @param array $packed
     * @param Dictionary $dict
     * @return array
     */
    protected static function unpackData(array $packed, Dictionary $dict): array
    {
        $name = null;
        $returns = null;
        $params = [];
        if (isset($packed[0])) {
            $packed[0] = self::unstringify($dict->read($packed[0]));
            $name = $packed[0][1] ?? null;
            $returns = isset($packed[0][2]) ? $dict->read($packed[0][2]) : null;
            if ($returns) {
                $returns = PhpTypeMeta::unpack($returns, $dict);
            }
            if (isset($packed[0][3]) && is_array($packed[0][3])) {
                foreach ($packed[0][3] as $param) {
                    $params[] = PhpParameterMeta::unpack($param, $dict);
                }
            }
        }
        $data = [];
        $data['name'] = $name;
        $data['returns'] = $returns;
        $data['params'] = $params ?: null;
        $data['checksum'] = $packed[1] ?? null;
        return $data;
    }

    /**
     * @param Dictionary $dict
     * @return array
     */
    protected function packData(Dictionary $dict): array
    {
        $data = [
            0 => [
                0 => static::getKind(),
                1 => $this->name,
                2 => $dict->write($this->returns->pack($dict)),
            ],
            1 => $this->checksum,
        ];
        if ($this->params) {
            foreach ($this->params as $param) {
                $data[0][3][] = $param->pack($dict);
            }
        }
        $data[0] = $dict->write(self::stringify($data[0]));
        return $data;
    }

    /**
     * @param mixed $data
     */
    public function __construct($data)
    {
        $this->name = $data['name'] ?? null;
        $this->returns = $data['returns'] ?? null;
        $this->params = isset($data['params']) ? array_map(function (PhpParameterMeta $param) {
            return $param;
        }, $data['params']) : null;
        $this->checksum = $data['checksum'] ?? null;
    }

    /**
     * @return string|null
     */
    public function getName(): ?string
    {
        return $this->name;
    }

    /**
     * @return Type|null
     */
    public function getReturnType(): ?Type
    {
        if ($this->returns) {
            return $this->returns->cast();
        }
        return null;
    }

    /**
     * @inheritDoc
     */
    public function getParams(): array
    {
        return $this->params ?? [];
    }

    /**
     * @return string|null
     */
    public function getImplementationChecksum(): ?string
    {
        return $this->checksum;
    }
}
