<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Data;

use Magento\Mray\CodeStructuralElement\Php\Reflection\ParameterDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\TypeFactory;
use Magento\Mray\Index\Collection\Dictionary;
use phpDocumentor\Reflection\Type;
use phpDocumentor\Reflection\Types;

class PhpParameterMeta extends DataTransferObject implements ParameterDeclaration
{
    private const VARIADIC = 1;
    private const OPTIONAL = 2;

    /** @var string  */
    protected $name;
    /** @var PhpTypeMeta  */
    protected $type;
    /** @var int  */
    protected $flags;

    /**
     * @param ParameterDeclaration $d
     * @return static
     */
    public static function createFromObject(ParameterDeclaration $d): self
    {
        $flags = 0;
        if ($d->isVariadic()) {
            $flags |= self::VARIADIC;
        }
        if ($d->isOptional()) {
            $flags |= self::OPTIONAL;
        }
        return new static([
            'name' => $d->getName(),
            'type' => PhpTypeMeta::createFromObject($d->getExpectedArgumentType()),
            'flags' => $flags,
        ]);
    }

    /**
     * @param array $packed
     * @param Dictionary $dict
     * @return array
     */
    protected static function unpackData(array $packed, Dictionary $dict): array
    {
        return [
            'name' => $packed[0] ?? null,
            'type' => isset($packed[1]) ? PhpTypeMeta::unpack($dict->read($packed[1]), $dict) : null,
            'flags' => $packed[2] ?? 0,
        ];
    }

    /**
     * @param Dictionary $dict
     * @return array
     */
    protected function packData(Dictionary $dict): array
    {
        return [
            0 => $this->name,
            1 => $dict->write($this->type->pack($dict)),
            2 => $this->flags ?: null,
        ];
    }

    /**
     * @param mixed $data
     */
    public function __construct($data)
    {
        $this->name = $data['name'] ?? null;
        $this->type = $data['type'] ?? null;
        $this->flags = $data['flags'] ?? 0;
    }

    /**
     * @return string|null
     */
    public function getName(): ?string
    {
        return $this->name;
    }

    /**
     * @return Type
     */
    public function getType(): Type
    {
        $argumentType = $this->getExpectedArgumentType();
        if ($this->isVariadic()) {
            return TypeFactory::get(Types\Array_::class, $argumentType);
        }
        return $argumentType;
    }

    /**
     * @return Type
     */
    public function getExpectedArgumentType(): Type
    {
        return $this->type->cast();
    }

    /**
     * @return bool
     */
    public function isVariadic(): bool
    {
        return (bool)($this->flags & self::VARIADIC);
    }

    /**
     * @return bool
     */
    public function isOptional(): bool
    {
        return (bool)($this->flags & self::OPTIONAL);
    }
}
