<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Data;

use Magento\Mray\CodeStructuralElement\Php\Reflection\Type\CodeDiscoveryDependentType;
use Magento\Mray\CodeStructuralElement\Php\Reflection\TypeFactory;
use Magento\Mray\Index\Collection\Dictionary;
use phpDocumentor\Reflection\Type;
use phpDocumentor\Reflection\Types;

class PhpTypeMeta extends DataTransferObject
{
    /** @var string  */
    protected $expr;
    /** @var Type  */
    private $resolvedType;

    /**
     * @param Type $t
     * @return static
     */
    public static function createFromObject(Type $t): self
    {
        return new static([
            'expr' => self::typeToExpression($t),
        ]);
    }

    /**
     * @param Type $t
     * @return string
     */
    private static function typeToExpression(Type $t): string
    {
        if ($t instanceof Types\Expression) {
            return self::typeToExpression($t->getValueType());
        }

        if ($t instanceof Types\Nullable) {
            $expr = self::typeToExpression($t->getActualType());
            $expr .= '|null';
            return $expr;
        }

        if ($t instanceof CodeDiscoveryDependentType) {
            return self::typeToExpression($t->getEstimatedType());
        }

        return (string)$t;
    }

    /**
     * @param array $packed
     * @param Dictionary $dict
     * @return array|string[]
     */
    protected static function unpackData(array $packed, Dictionary $dict): array
    {
        return [
            'expr' => $packed[0] ?? 'mixed',
        ];
    }

    /**
     * @param Dictionary $dict
     * @return array|null[]|string[]
     */
    protected function packData(Dictionary $dict): array
    {
        return [
            0 => $this->expr === 'mixed' ? null : $this->expr,
        ];
    }

    /**
     * @param mixed $data
     */
    public function __construct($data)
    {
        $this->expr = $data['expr'] ?? 'mixed';
    }

    /**
     * @return Type
     */
    public function cast(): Type
    {
        if (!isset($this->resolvedType)) {
            $this->resolvedType = TypeFactory::cast($this->expr);
        }
        return $this->resolvedType;
    }
}
