<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index;

use Magento\Mray\CodeStructuralElement\Php\Reflection\Context\StructuralElementsKnowledgeBase;
use Magento\Mray\MagentoApiIndex\Index\Version;
use Magento\Mray\Index\Data\PhpTypeMeta;
use Magento\Mray\Index\Data\PhpParameterMeta;
use function array_filter;
use function array_map;

class MagentoApiIndexAdapter implements StructuralElementsKnowledgeBase
{
    /** @var Version */
    private $version;

    /** @var array */
    private $cache;

    /**
     * @param Version $version
     */
    public function __construct(Version $version)
    {
        $this->version = $version;
    }

    /**
     * @inheritDoc
     */
    public function findClassLike(string $name): array
    {
        if (isset($this->cache[$name])) {
            return $this->cache[$name];
        }

        $data = $this->version->describe($name);
        if (!$data) {
            return $this->cache[$name] = [];
        }

        if (isset($data['members'])) {
            $data['members'] = array_filter(array_map(
                'self::getPhpMetaData',
                $data['members']
            ));
        }

        switch ($data['kind']) {
            case 'class':
                return  $this->cache[$name] = [ new PhpClassMeta($data) ];
            case 'interface':
                return $this->cache[$name] = [ new PhpInterfaceMeta($data) ];
            default:
                return $this->cache[$name] = [];
        }
    }

    /**
     * @param array $mData
     * @return PhpClassConstantMeta|PhpMethodMeta|PhpPropertyMeta|null
     */
    public static function getPhpMetaData(array $mData)
    {
        switch ($mData['kind'] ?? null) {
            case 'method':
                if (isset($mData['returns'])) {
                    $mData['returns'] = new PhpTypeMeta($mData['returns']);
                }
                if (isset($mData['params'])) {
                    $mData['params'] = array_map(static function (array $pData) {
                        if (isset($pData['type'])) {
                            $pData['type'] = new PhpTypeMeta($pData['type']);
                        }
                        return new PhpParameterMeta($pData);
                    }, $mData['params']);
                }
                return new PhpMethodMeta($mData);
            case 'property':
                if (isset($mData['type'])) {
                    $mData['type'] = new PhpTypeMeta($mData['type']);
                }
                return new PhpPropertyMeta($mData);
            case 'constant':
                if (isset($mData['type'])) {
                    $mData['type'] = new PhpTypeMeta($mData['type']);
                }
                return new PhpClassConstantMeta($mData);
            default:
                return null;
        }
    }

    /**
     * @inheritDoc
     */
    public function findFunction(string $name): array
    {
        // todo: add support of global functions
        return [];
    }

    /**
     * @inheritDoc
     */
    public function findGlobalConstant(string $name): array
    {
        // todo: add support of global constants
        return [];
    }
}
