<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Persistence;

use RuntimeException;
use function clearstatcache;
use function dirname;
use function fclose;
use function fopen;
use function fwrite;
use function get_object_vars;
use function is_array;
use function is_dir;
use function is_null;
use function is_object;
use function is_scalar;
use function mkdir;
use function sprintf;
use function var_export;

class FileWriter
{
    /**
     * @param string $file
     * @param mixed $data
     */
    public function write(string $file, $data): void
    {
        $indexFileDir = dirname($file);
        if (!is_dir($indexFileDir)) {
            mkdir($indexFileDir, 0755, true);
            clearstatcache(true, $indexFileDir);
        }
        // phpcs:ignore Generic.PHP.NoSilencedErrors
        $f = @fopen($file, 'w');
        if ($f === false) {
            throw new RuntimeException(sprintf('Unable to write index data into %s.', $file));
        }
        try {
            fwrite($f, '<?php return ');
            foreach ($this->print($data) as $token) {
                fwrite($f, $token);
            }
            fwrite($f, ';');
        } finally {
            fclose($f);
        }
    }

    /**
     * @param mixed $data
     * @return iterable
     */
    private function print($data): iterable
    {
        if (is_scalar($data) || is_null($data)) {
            yield var_export($data, true);
            return;
        }

        if (is_object($data)) {
            yield '(object)';
            $data = get_object_vars($data);
        }

        if (is_array($data)) {
            yield '[';
            $i = 0;
            $list = true;
            foreach ($data as $key => $val) {
                $list = $list && $key === $i;

                if ($i) {
                    yield ',';
                }
                if (!$list) {
                    yield from $this->print($key);
                    yield '=>';
                }
                yield from $this->print($val);
                $i++;
            }
            yield ']';
        } else {
            yield var_export($data, true);
        }
    }
}
