<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Persistence;

use Magento\Mray\Index\Collection\MagentoApi;
use Magento\Mray\MagentoApiIndex\Api\GetIndexedVersions;
use Magento\Mray\MagentoApiIndex\Index\Read;
use Magento\Mray\MagentoApiIndex\Model\Volumes;
use Magento\Mray\MagentoApiIndex\Model\Versions;
use Magento\Mray\Index\Persistence\ReferenceUpdate;
use function clearstatcache;
use function dirname;
use function is_dir;
use function mkdir;
use function realpath;
use function sprintf;

class MagentoApiStorage
{
    /**
     * @var string
     */
    private $dir;

    /**
     * @var Volumes
     */
    private $volumes;

    /**
     * @var ReferenceUpdate
     */
    private $referenceUpdate;

    /**
     * @param string $dataDir
     */
    public function __construct(string $dataDir)
    {
        $dir = $dataDir . '/api';
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
            clearstatcache(true, $dir);
        }

        $this->dir = realpath($dir);
        $this->volumes = new Volumes();
        $this->referenceUpdate = new ReferenceUpdate(new Versions(new GetIndexedVersions(new Read($dataDir))));
    }

    /**
     * @param MagentoApi $api
     * @param string $version
     */
    public function append(MagentoApi $api, string $version)
    {
        $writer = new FileWriter();

        foreach ($this->volumes($api, $version) as $volume => $data) {
            $file = sprintf('%s/%s.php', $this->dir, $volume);
            $writer->write($file, $data);
        }
    }

    /**
     * @param MagentoApi $api
     * @param string $version
     * @return array
     */
    private function volumes(MagentoApi $api, string $version): array
    {
        $reader = new FileReader();
        $releases = new MagentoReleaseStorage(dirname($this->dir));

        $volumes = [];

        foreach ($api->exportData() as $reference => $value) {
            $volume = $this->volumes->getVolumeForReference($reference);
            if ($volume === '') {
                continue;
            }

            if (!isset($volumes[$volume])) {
                $file = sprintf('%s/%s.php', $this->dir, $volume);
                $volumes[$volume] = $reader->read($file);
            }

            // $volumes[$volume][$reference] = ['api' => '2.4.3', 'deprecated' => '2.4.4,2.4.5']
            if (isset($volumes[$volume][$reference])) {
                $volumes[$volume][$reference] = $this->referenceUpdate->execute(
                    $volumes[$volume][$reference],
                    $value,
                    $version
                );
                continue;
            }
            $knownMask = $volumes[$volume][$reference][$value] ?? '';
            $volumes[$volume][$reference][$value] = $releases->mask($version, $knownMask);
        }
        return $volumes;
    }
}
