<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Persistence;

use Magento\Mray\Index\Collection\PhpCode;
use Magento\Mray\MagentoApiIndex\Api\GetIndexedVersions;
use Magento\Mray\MagentoApiIndex\Index\Read;
use Magento\Mray\MagentoApiIndex\Model\Volumes;
use Magento\Mray\MagentoApiIndex\Model\Versions;
use function clearstatcache;
use function dirname;
use function is_dir;
use function mkdir;
use function realpath;
use function sprintf;

class PhpCodeStorage
{
    /**
     * @var string
     */
    private $dir;

    /**
     * @var Volumes
     */
    private $volumes;

    /**
     * @var ReferenceUpdate
     */
    private $referenceUpdate;

    /**
     * @param string $dataDir
     */
    public function __construct(string $dataDir)
    {
        $dir = $dataDir . '/phpCode';
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
            clearstatcache(true, $dir);
        }

        $this->dir = realpath($dir);
        $this->volumes = new Volumes();
        $this->referenceUpdate = new ReferenceUpdate(new Versions(new GetIndexedVersions(new Read($dataDir))));
    }

    /**
     * @param PhpCode $phpCode
     * @param string $version
     */
    public function append(PhpCode $phpCode, string $version)
    {
        $writer = new FileWriter();

        foreach ($this->volumes($phpCode, $version) as $volume => $data) {
            $file = sprintf('%s/%s.php', $this->dir, $volume);
            $writer->write($file, $data);
        }
    }

    /**
     * @param PhpCode $phpCode
     * @param string $version
     * @return array
     */
    private function volumes(PhpCode $phpCode, string $version): array
    {
        $reader = new FileReader();
        $releases = new MagentoReleaseStorage(dirname($this->dir));

        $volumes = [];
        foreach ($phpCode->exportData() as $reference => $value) {
            $volume = $this->volumes->getVolumeForReference($reference);
            if ($volume === '') {
                continue;
            }

            if (!isset($volumes[$volume])) {
                $file = sprintf('%s/%s.php', $this->dir, $volume);
                $volumes[$volume] = $reader->read($file);
            }

            foreach ($value as $impl) {
                if (isset($volumes[$volume][$reference])) {
                    $volumes[$volume][$reference] = $this->referenceUpdate->execute(
                        $volumes[$volume][$reference],
                        $impl,
                        $version
                    );
                    continue;
                }
                $knownMask = $volumes[$volume][$reference][$impl] ?? '';
                $volumes[$volume][$reference][$impl] = $releases->mask($version, $knownMask);
            }
        }
        return $volumes;
    }
}
