<?php
/**
 * Copyright 2022 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Persistence;

use Magento\Mray\MagentoApiIndex\Model\Versions;

class ReferenceUpdate
{
    /**
     * @var Versions
     */
    private $versions;

    /**
     * @param Versions|null $versions
     */
    public function __construct(Versions $versions = null)
    {
        $this->versions = $versions ?? new Versions();
    }

    /**
     * @param array $referenceArray
     * @param string|int $newValue
     * @param string $version
     * @return array
     */
    public function execute(array $referenceArray, $newValue, string $version): array
    {
        if (!isset($referenceArray[$newValue])) {
            // Scenario 1: There is a new dictioanry reference for current version.
            // Because for single version there is only one dictionary entry,
            // first we need to find and REMOVE
            // existing current version occurence from dictionary references [1]
            // Then, we add new dictionary reference with current version [2]
            // Also, there is a case when old dictionary reference was used ONLY for current
            // version and after removing it from mask its value becomes 0. In that case we simply remove
            // dictionary reference [3]:
            foreach ($referenceArray as $reference => $mask) {
                if ($this->versions->isInMask($version, $mask)) {
                    $referenceArray = $this->removeFromMask($referenceArray, $reference, $version, $mask); // [1]
                }
            }
            $referenceArray[$newValue] = $this->versions->getMask([$version]); // [2]
            return $referenceArray;
        }

        $knownMask = $referenceArray[$newValue] ?? '';

        // Scenario 2: Entry exists, but there is other dictionary reference used for
        // current version. In this case, we have to first remove current version from other
        // dictionary reference [1], then ADD current version to the bitmask of existing dictionary
        // reference [2].
        foreach ($referenceArray as $value => $mask) {
            if ($value !== $newValue && $this->versions->isInMask($version, $mask)) {
                $referenceArray = $this->removeFromMask($referenceArray, $value, $version, $mask); // [1]
            }
            $referenceArray[$newValue] = $this->versions->addToMask($version, $knownMask); // [2]
        }
        return $referenceArray;
    }

    /**
     * @param array $referenceArray
     * @param string|int $value
     * @param string $version
     * @param string $mask
     * @return array
     */
    private function removeFromMask(array $referenceArray, $value, string $version, string $mask): array
    {
        $updatedMask = $this->versions->removeFromMask($version, $mask); // [1]
        if ($updatedMask === "0") {
            unset($referenceArray[$value]); // [3]
        } else {
            $referenceArray[$value] = $updatedMask;
        }
        return $referenceArray;
    }
}
