<?php
/**
 * Copyright 2022 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Persistence;

use Magento\Mray\Index\Collection\VirtualTypes;
use Magento\Mray\MagentoApiIndex\Model\Volumes;
use function clearstatcache;
use function dirname;
use function is_dir;
use function mkdir;
use function realpath;
use function sprintf;

class VirtualTypesStorage
{
    /**
     * @var string
     */
    private $dir;

    /**
     * @var Volumes
     */
    private $volumes;

    /**
     * @param string $dataDir
     */
    public function __construct(string $dataDir)
    {
        $dir = $dataDir . '/virtualTypes';
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
            clearstatcache(true, $dir);
        }

        $this->dir = realpath($dir);
        $this->volumes = new Volumes();
    }

    /**
     * @param VirtualTypes $virtualTypes
     * @param string $version
     */
    public function append(VirtualTypes $virtualTypes, string $version)
    {
        $writer = new FileWriter();

        foreach ($this->volumes($virtualTypes, $version) as $volume => $data) {
            $file = sprintf('%s/%s.php', $this->dir, $volume);
            $writer->write($file, $data);
        }
    }

    /**
     * @param VirtualTypes $virtualTypes
     * @param string $version
     * @return array
     */
    private function volumes(VirtualTypes $virtualTypes, string $version): array
    {
        $reader = new FileReader();
        $releases = new MagentoReleaseStorage(dirname($this->dir));

        $volumes = [];

        foreach ($virtualTypes->exportData() as $value) {
            $volume = $this->volumes->getVolumeForReference($value);
            if ($volume === '') {
                continue;
            }

            if (!isset($volumes[$volume])) {
                $file = sprintf('%s/%s.php', $this->dir, $volume);
                $volumes[$volume] = $reader->read($file);
            }

            $knownMask = $volumes[$volume][$value] ?? '';
            $volumes[$volume][$value] = $releases->mask($version, $knownMask);
        }
        return $volumes;
    }
}
