<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index;

use Magento\Mray\CodeStructuralElement\Php\Reflection\InterfaceDeclaration;
use Magento\Mray\Index\Collection\Dictionary;
use Magento\Mray\Index\Data\PhpClassLikeMeta;
use function array_map;

class PhpInterfaceMeta extends PhpClassLikeMeta implements InterfaceDeclaration
{
    /**
     * @param InterfaceDeclaration $d
     * @return static
     */
    public static function createFromObject(InterfaceDeclaration $d): self
    {
        $data = self::dataFromObject($d);

        $data['extends'] = $d->getExtendedInterfaces();

        return new static($data);
    }

    /**
     * @return int
     */
    public static function getKind(): int
    {
        return PhpElementKinds::DECLARED_ | PhpElementKinds::INTERFACE_;
    }

    /**
     * @param array $packed
     * @param Dictionary $dict
     * @return array
     */
    protected static function unpackData(array $packed, Dictionary $dict): array
    {
        if (isset($packed[0])) {
            $packed[0] = self::unstringify($dict->read($packed[0]));
        }

        $extends = [];
        if (isset($packed[11])) {
            foreach ($packed[11] as $extend) {
                $extends[] = $dict->read($extend);
            }
        }

        $data = parent::unpackData($packed, $dict);
        $data['extends'] = $extends ?: null;
        return $data;
    }

    /**
     * @param Dictionary $dict
     * @return array
     */
    protected function packData(Dictionary $dict): array
    {
        $data = parent::packData($dict);

        // inheritance information moved to top level as contract described by members signatures
        if ($this->extends) {
            foreach ($this->extends as $extends) {
                $data[11][] = $dict->write($extends); // no reason but consistent with class
            }
        }

        $data[0] = $dict->write(self::stringify($data[0]));
        return $data;
    }

    /** @var string[]|null  */
    protected $extends;

    /**
     * @param array $data
     */
    public function __construct($data)
    {
        parent::__construct($data);

        $this->extends = isset($data['extends']) ? array_map(function (string $n) {
            return $n;
        }, $data['extends']): null;
    }

    /**
     * @inheritDoc
     */
    public function getImplementationProviders(): array
    {
        return $this->getExtendedInterfaces();
    }

    /**
     * @return array|string[]
     */
    public function getExtendedInterfaces(): array
    {
        return $this->extends ?? [];
    }
}
