<?php
/**
 * Copyright 2022 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Release;

use Exception;
use Magento\Mray\Api;
use Magento\Mray\Index\Persistence\AppendIndex;
use Symfony\Component\Console\Output\OutputInterface;
use function basename;
use function count;
use function date;
use function file_exists;
use function file_get_contents;
use function file_put_contents;
use function getmypid;
use function key;
use function pcntl_fork;
use function pcntl_waitpid;
use function pcntl_wexitstatus;
use function pcntl_wifexited;
use function reset;
use function sleep;
use function sprintf;
use function str_repeat;
use function unlink;
use const DATE_ATOM;

class ParallelIndex
{
    /**
     * @var OutputInterface
     */
    private $output;

    /**
     * @param OutputInterface $output
     */
    public function __construct(OutputInterface $output)
    {
        $this->output = $output;
    }

    /**
     * @param array $paths
     * @param string $outputDirectory
     * @return int
     */
    public function execute(array $paths, string $outputDirectory): int
    {
        if (count($paths) === 1) {
            $path = reset($paths);
            $relativePath = basename($path);
            try {
                $this->singleProjectIndex($path, $outputDirectory);
                // phpcs:ignore Magento2.Security.LanguageConstruct
                return 0;
            } catch (Exception $e) {
                $this->outputError(
                    sprintf('✗ Failed indexation of %s: %s', $relativePath, $e->getMessage())
                );
                // phpcs:ignore Magento2.Security.LanguageConstruct
                return 1;
            }
        }

        return $this->multiprocessIndex($paths, $outputDirectory) ? 1 : 0;
    }

    /**
     * @param string $path
     * @param string $outputDirectory
     * @param string|null $semaphore
     */
    private function singleProjectIndex(string $path, string $outputDirectory, string $semaphore = null): void
    {
        $relativePath = basename($path);
        $this->output->writeln(
            str_repeat(' ', 27).
            sprintf('↳ [%s] Building indexes...', $relativePath)
        );

        $indexResult = (new Api\MagentoReleaseIndex())->execute($path);

        if ($semaphore) {
            while ((int)file_get_contents($semaphore) != getmypid()) {
                sleep(5);
            }
        }

        $this->output->writeln(
            str_repeat(' ', 27).
            sprintf('↳ [%s] Writing to filesystem...', $relativePath)
        );
        (new AppendIndex())->execute($indexResult, $outputDirectory);
    }

    /**
     * @param array $paths
     * @param string $outputDirectory
     * @return bool
     */
    private function multiprocessIndex(array $paths, string $outputDirectory): bool
    {
        $error = false;
        $processIds = [];
        $semaphore = $outputDirectory . '/lock';
        foreach ($paths as $path) {
            $relativePath = basename($path);
            $this->output->writeln(sprintf('[%s] Processing %s...', date(DATE_ATOM), $relativePath));
            switch ($processId = pcntl_fork()) {
                case -1:
                    $this->outputError(
                        sprintf('✗ Failed indexation of %s: unable to start indexation thread.', $relativePath)
                    );
                    $error = true;
                    break;
                case 0:
                    try {
                        $this->singleProjectIndex($path, $outputDirectory, $semaphore);
                        // phpcs:ignore Magento2.Security.LanguageConstruct
                        exit(0); // exit child process to avoid loop in it
                    } catch (Exception $e) {
                        $this->outputError(
                            sprintf('✗ Failed indexation of %s: %s', $relativePath, $e->getMessage())
                        );
                        // phpcs:ignore Magento2.Security.LanguageConstruct
                        exit(1); // exit child process to avoid loop in it
                    }
                // no break
                default:
                    $processIds[$processId] = $relativePath;

                    reset($processIds);
                    $firstPid = key($processIds);
                    if (count($processIds) >= 4) {
                        pcntl_waitpid($firstPid, $status);
                        if (!pcntl_wifexited($status)) {
                            $this->outputError(
                                sprintf('✗ [%s] Failed: indexation thread terminated.', $processIds[$firstPid])
                            );
                            $error = true;
                        } elseif (pcntl_wexitstatus($status) !== 0) {
                            $this->outputError(
                                sprintf('✗ [%s] Failed: indexation thread not succeed.', $processIds[$firstPid])
                            );
                            $error = true;
                        }
                        unset($processIds[$firstPid]);
                        reset($processIds);
                        $firstPid = key($processIds);
                    }
                    file_put_contents($semaphore, $firstPid);

                    break;
            }
        }

        while (count($processIds)) {
            reset($processIds);
            $firstPid = key($processIds);
            pcntl_waitpid($firstPid, $status);
            if (!pcntl_wifexited($status)) {
                $this->outputError(sprintf('✗ [%s] Failed: indexation thread terminated.', $processIds[$firstPid]));
                $error = true;
            } elseif (pcntl_wexitstatus($status) !== 0) {
                $this->outputError(sprintf('✗ [%s] Failed: indexation thread not succeed.', $processIds[$firstPid]));
                $error = true;
            }
            unset($processIds[$firstPid]);

            reset($processIds);
            $firstPid = key($processIds);
            file_put_contents($semaphore, $firstPid);
        }
        if (file_exists($semaphore)) {
            unlink($semaphore);
        }

        return $error;
    }

    /**
     * @param string $message
     */
    private function outputError(string $message): void
    {
        $this->output->getErrorStyle()->writeln(str_repeat(' ', 27) . $message);
    }
}
