<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Scanner;

use Closure;
use Magento\Mray\Package\AbstractTree\Node\Component;
use Magento\Mray\Package\AbstractTree\ComponentHolder;
use Magento\Mray\Package\AbstractTree\Node\DistributionPackage;
use Magento\Mray\Package\AbstractTree\Node\Package;
use Magento\Mray\Package\AbstractTree\Scanner;
use Magento\Mray\Package\AbstractTree\ScannerSubject;
use function array_filter;
use function array_map;

class GeneralPurposeScanner implements Scanner
{
    /** @var Probe[] */
    private $probes;
    /** @var Tuner  */
    private $tuner;

    /**
     * @param Probe[] $probes
     * @param Tuner $tuner
     */
    public function __construct(array $probes, Tuner $tuner)
    {
        $this->probes = array_map(function (Probe $p) {
            return $p;
        }, $probes);
        $this->tuner = $tuner;
    }

    /**
     * @inheritDoc
     */
    public function scan(ScannerSubject $subject): iterable
    {
        list($packages, $nestedPackagesCanExist) = $this->findPackages($subject);

        if (!$this->proceedScan($nestedPackagesCanExist)) {
            yield from $packages;
            return;
        }

        if (empty($packages)) {
            yield from $this->scanFragments($subject);
            return;
        }

        $registerNestedComponent = $this->createNestedPackageRegistrationStrategy($packages);
        foreach ($this->scanFragments($subject) as $nestedPackage) {
            if ($nestedPackage instanceof DistributionPackage) {
                $packages[] = $nestedPackage;
            } else {
                $registerNestedComponent($nestedPackage);
            }
        }

        yield from $packages;
    }

    /**
     * @param ScannerSubject $subject
     * @return array
     * @throws FailedAssumption
     */
    private function findPackages(ScannerSubject $subject): array
    {
        /** @var Package[] $packages */
        $packages = [];
        $nestedPackagesCanExist = Sample::NESTED_PACKAGES_MAYBE;
        foreach ($this->probes as $probe) {
            $sample = $probe->check($subject);
            if ($sample) {
                if ($sample->package()) {
                    $packages[] = $sample->package();
                }
                $nestedPackagesCanExist |= $sample->nestedPackagesCanExist();
            }
        }
        $packages = $this->tuner->tune($subject, $packages);
        $packages = $this->compactPackages($packages);

        return [$packages, $nestedPackagesCanExist];
    }

    /**
     * @param array $packages
     * @return Package[]
     */
    private function compactPackages(array $packages)
    {
        /**
         * @var DistributionPackage[] $distributionPackages
         * @var Component[]|ComponentHolder[] $topLevelComponents
         * @var Component[] $integralComponents
         */
        $distributionPackages = [];
        $compoundComponents = [];
        $integralComponents = [];

        foreach ($packages as $package) {
            if ($package instanceof DistributionPackage) {
                $distributionPackages[] = $package;
            } elseif ($package instanceof Component) {
                if ($package instanceof ComponentHolder) {
                    $compoundComponents[] = $package;
                } else {
                    $integralComponents[] = $package;
                }
            }
        }

        $topLevelComponents = [];
        foreach ($compoundComponents as $topLevelComponent) {
            $topLevelComponent->registerComponents($integralComponents);
            $topLevelComponents[] = $topLevelComponent;
        }
        if (empty($topLevelComponents)) {
            $topLevelComponents = $integralComponents;
        }

        if (empty($distributionPackages)) {
            return $topLevelComponents;
        }

        if (!empty($topLevelComponents)) {
            foreach ($distributionPackages as $distributionPackage) {
                $distributionPackage->markAsComponent();
                $distributionPackage->registerComponents($topLevelComponents);
            }
        }

        return $distributionPackages;
    }

    /**
     * @param int $instruction
     * @return bool
     */
    private function proceedScan(int $instruction): bool
    {
        if (($instruction & Sample::NESTED_PACKAGES_EXPECTED) || $instruction === Sample::NESTED_PACKAGES_MAYBE) {
            return true;
        }
        return false;
    }

    /**
     * @param array $packages
     * @return Closure
     */
    private function createNestedPackageRegistrationStrategy(array &$packages): Closure
    {
        $topLevelPackages = [];
        foreach (array_filter($packages, function ($p) {
            return $p instanceof ComponentHolder;
        }) as $topLevelPackage) {
            if ($topLevelPackage instanceof DistributionPackage && $topLevelPackage->isComponent()) {
                $containers = array_filter($topLevelPackage->components, function ($p) {
                    return $p instanceof ComponentHolder;
                });
            }
            if (!empty($containers)) {
                $topLevelPackages += $containers;
            } else {
                $topLevelPackages[] = $topLevelPackage;
            }
            unset($topLevelPackage, $containers);
        }

        if (empty($topLevelPackages)) {
            return function (Component $c) use (&$packages) {
                $packages[] = $c;
            };
        } else {
            return function (Component $c) use ($topLevelPackages) {
                foreach ($topLevelPackages as $package) {
                    $package->registerComponent($c);
                    if ($package instanceof DistributionPackage) {
                        $package->markAsContainer();
                    }
                }
            };
        }
    }

    /**
     * @param ScannerSubject $subject
     * @return iterable
     */
    private function scanFragments(ScannerSubject $subject): iterable
    {
        foreach ($subject->fragments() as $fragment) {
            yield from $this->scan($fragment);
        }
        return;
        // phpcs:ignore Squiz.PHP.NonExecutableCode
        yield; // do not remove, yield after return lead to empty generator
    }
}
