<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Scanner\Probe;

use Magento\Mray\Package\AbstractTree\DistributionPackage;
use Magento\Mray\Package\AbstractTree\Node\Identifier;
use Magento\Mray\Package\AbstractTree\Identifier\Versioned;
use Magento\Mray\Package\AbstractTree\Identifier\ComposerPackage;
use Magento\Mray\Index\Scanner\FailedAssumption;
use Magento\Mray\Index\Scanner\Probe;
use Magento\Mray\Index\Scanner\Sample;
use Magento\Mray\Package\AbstractTree\ScannerSubject;
use Magento\Mray\Package\AbstractTree\Node\Version;
use Magento\Mray\Package\AbstractTree\Version\SemVerLike;
use Magento\Mray\Package\AbstractTree\Version\Named;
use Throwable;
use function array_keys;
use function count;
use function explode;
use function preg_match;
use function sprintf;
use function strtolower;

class ComposerDistributionPackage implements Probe
{
    private const COMPOSER_JSON_FILE = 'composer.json';

    /** @var SubjectReader\Json  */
    private $reader;

    /**
     * ComposerDistributionPackage constructor.
     */
    public function __construct()
    {
        $this->reader = new SubjectReader\Json();
    }

    /**
     * @param ScannerSubject $subject
     * @return Sample|null
     * @throws FailedAssumption
     */
    public function check(ScannerSubject $subject): ?Sample
    {
        if (!$subject->contains(self::COMPOSER_JSON_FILE)) {
            return null;
        }

        try {
            $composerData = $subject->read(self::COMPOSER_JSON_FILE, $this->reader);
            if (!isset($composerData['name'])) {
                return null;
            }
            $identifier = $this->parseIdentifier($composerData['name']);
            if (!$identifier) {
                return null;
            }
            if (isset($composerData['version'])) {
                $identifier = new Versioned(
                    $identifier,
                    $this->parseVersion($composerData['version'])
                );
            }
            $namespaces =
                isset($composerData['autoload']['psr-4']) ? array_keys($composerData['autoload']['psr-4']) : [];
            return new Sample(new DistributionPackage\Composer(
                $identifier,
                $composerData['type'] ?? 'library',
                $namespaces
            ));
        } catch (Throwable $e) {
            throw new FailedAssumption(
                sprintf(
                    'Invalid %s file in %s: %s',
                    self::COMPOSER_JSON_FILE,
                    $subject->location(),
                    $e->getMessage()
                ),
                $e->getCode(),
                $e
            );
        }
    }

    /**
     * @param mixed $name
     * @return Identifier|null
     * @throws FailedAssumption
     */
    private function parseIdentifier($name): ?Identifier
    {
        $name = (string)$name;
        $parts = explode('/', (string)$name, 2);
        if (count($parts) !== 2) {
            return null;
        }

        return new ComposerPackage($parts[1], $parts[0]);
    }

    /**
     * @param mixed $version
     * @return Version
     */
    private function parseVersion($version): Version
    {
        $version = strtolower((string)$version);
        $likeSemVer = preg_match(
            '/^(?P<prefix>\S*?)(?P<major>0|[1-9]\d*)\.(?P<minor>0|[1-9]\d*)\.(?P<patch>0|[1-9]\d*)(?P<suffix>-?\S*)$/',
            $version,
            $semVer
        );
        if ($likeSemVer) {
            return new SemVerLike(
                (int)$semVer['major'],
                (int)$semVer['minor'],
                (int)$semVer['patch'],
                $semVer['suffix'],
                $semVer['prefix']
            );
        }
        return new Named($version);
    }
}
