<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Scanner\Probe;

use Magento\Mray\Package\AbstractTree\Component;
use Magento\Mray\Package\AbstractTree\Identifier\Named;
use Magento\Mray\Package\AbstractTree\Identifier\Unidentifiable;
use Magento\Mray\Package\AbstractTree\Node\Identifier;
use Magento\Mray\Index\Scanner\FailedAssumption;
use Magento\Mray\Index\Scanner\Probe;
use Magento\Mray\Index\Scanner\Sample;
use Magento\Mray\Package\AbstractTree\ScannerSubject;
use Throwable;
use function array_filter;
use function array_values;
use function sprintf;
use function strtolower;

class MagentoLibrary implements Probe
{
    private const COMPOSER_JSON_FILE = 'composer.json';
    private const REGISTRATION_FILE = 'registration.php';

    /** @var SubjectReader\Json  */
    private $jsonReader;
    /** @var SubjectReader\MagentoComponentRegistration  */
    private $regReader;

    /**
     * MagentoLibrary constructor.
     */
    public function __construct()
    {
        $this->jsonReader = new SubjectReader\Json();
        $this->regReader = new SubjectReader\MagentoComponentRegistration();
    }

    /**
     * @inheritDoc
     */
    public function check(ScannerSubject $subject): ?Sample
    {
        try {
            if (!$subject->contains(self::COMPOSER_JSON_FILE)) {
                return null;
            }
            $composerData = $subject->read(self::COMPOSER_JSON_FILE, $this->jsonReader);
            if (isset($composerData['type']) && strtolower((string)$composerData['type']) !=='magento2-library') {
                return null;
            }
        } catch (Throwable $e) {
            throw new FailedAssumption(
                sprintf(
                    'Invalid %s file at %s: %s',
                    self::COMPOSER_JSON_FILE,
                    $subject->location(),
                    $e->getMessage()
                ),
                $e->getCode(),
                $e
            );
        }
        try {
            if (!$subject->contains(self::REGISTRATION_FILE)) {
                return null;
            }
            $regs = array_values(array_filter(
                $subject->read(self::REGISTRATION_FILE, $this->regReader),
                function (array $reg) {
                    return
                        $reg[SubjectReader\MagentoComponentRegistration::TYPE] ===
                        SubjectReader\MagentoComponentRegistration::TYPE_LIBRARY;
                }
            ));
        } catch (Throwable $e) {
            throw new FailedAssumption(
                sprintf('Invalid %s file at %s: %s', self::REGISTRATION_FILE, $subject->location(), $e->getMessage()),
                $e->getCode(),
                $e
            );
        }

        if (count($regs) > 1) {
            throw new FailedAssumption(sprintf(
                'Exact one declaration of Magento library is expected in %s at %s.',
                self::REGISTRATION_FILE,
                $subject->location()
            ));
        }

        return new Sample(
            new Component\MagentoLibrary(
                $this->readIdentifier($regs[0][SubjectReader\MagentoComponentRegistration::NAME] ?? null)
            ),
            Sample::NESTED_PACKAGES_NONE
        );
    }

    /**
     * @param mixed $name
     * @return Identifier
     */
    private function readIdentifier($name): Identifier
    {
        $name = trim((string)$name);
        if ($name) {
            return new Named($name);
        } else {
            return new Unidentifiable();
        }
    }
}
