<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Scanner\Probe;

use Magento\Mray\Package\AbstractTree\Component;
use Magento\Mray\Package\AbstractTree\Identifier;
use Magento\Mray\Index\Scanner\FailedAssumption;
use Magento\Mray\Index\Scanner\Probe;
use Magento\Mray\Index\Scanner\Sample;
use Magento\Mray\Package\AbstractTree\ScannerSubject;
use SimpleXMLElement;
use Throwable;
use function explode;
use function sprintf;

class MagentoModule implements Probe
{
    private const MODULE_XML_FILE = 'etc/module.xml';

    /** @var SubjectReader\SimpleXml  */
    private $reader;

    /**
     * MagentoModule constructor.
     */
    public function __construct()
    {
        $this->reader = new SubjectReader\SimpleXml();
    }

    /**
     * @inheritDoc
     */
    public function check(ScannerSubject $subject): ?Sample
    {
        if (!$subject->contains(self::MODULE_XML_FILE)) {
            return null;
        }

        try {
            $xml = $subject->read(self::MODULE_XML_FILE, $this->reader);
        } catch (Throwable $e) {
            return null;
        }

        if (!isset($xml->module) || !$xml->module instanceof SimpleXMLElement) {
            return null;
        }

        return new Sample(
            new Component\MagentoModule($this->parseIdentifier((string)$xml->module['name'])),
            Sample::NESTED_PACKAGES_NONE
        );
    }

    /**
     * @param string $name
     * @return Identifier\MagentoModule|Identifier\Named
     */
    private function parseIdentifier(string $name)
    {
        $parts = explode('_', $name, 2);
        if (count($parts) !== 2) {
            return new Identifier\Named($name);
        }

        return new Identifier\MagentoModule($parts[1], $parts[0]);
    }
}
