<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Scanner\Probe;

use Magento\Mray\Package\AbstractTree\Component;
use Magento\Mray\Package\AbstractTree\Identifier;
use Magento\Mray\Index\Scanner\FailedAssumption;
use Magento\Mray\Index\Scanner\Probe;
use Magento\Mray\Index\Scanner\Sample;
use Magento\Mray\Package\AbstractTree\ScannerSubject;
use Throwable;
use function array_intersect;
use function array_keys;
use function is_array;

class MagentoProject implements Probe
{
    private const COMPOSER_JSON_FILE = 'composer.json';
    private const COMPOSER_LOCK_FILE = 'composer.lock';

    /**
     * @var SubjectReader\Json
     */
    private $reader;

    /**
     * MagentoProject constructor.
     */
    public function __construct()
    {
        $this->reader = new SubjectReader\Json();
    }

    /**
     * @inheritDoc
     */
    public function check(ScannerSubject $subject): ?Sample
    {
        if (!$subject->contains(self::COMPOSER_JSON_FILE)) {
            return null;
        }

        if (!$subject->contains(self::COMPOSER_LOCK_FILE)) {
            return null;
        }

        try {
            $composerData = $subject->read(self::COMPOSER_JSON_FILE, $this->reader);
        } catch (Throwable $e) {
            throw new FailedAssumption(
                $e->getMessage(),
                $e->getCode(),
                $e
            );
        }

        if ($this->includesMagentoCore($composerData)) {
            return new Sample(new Component\MagentoProject(new Identifier\Unidentifiable()));
        }

        if (isset($composerData['replace']['magento/framework'])) {
            $projectName = isset($composerData['name'])
                ? new Identifier\Named($composerData['name'])
                : new Identifier\Unidentifiable();
            return new Sample(new Component\MagentoProject($projectName));
        }

        return null;
    }

    /**
     * @param mixed $composerData
     * @return bool
     * @throws FailedAssumption
     */
    private function includesMagentoCore($composerData): bool
    {
        if (!isset($composerData['require'])) {
            return false;
        }

        if (!is_array($composerData['require'])) {
            throw new FailedAssumption('Expected data structure as value of "require" section.');
        }

        $core = array_intersect(
            array_keys($composerData['require']),
            [
                'magento/product-enterprise-edition',
                'magento/magento2-ee-base',
                'magento/product-community-edition',
                'magento/magento2-base',
                'magento/magento-cloud-metapackage',
            ]
        );

        return !empty($core);
    }
}
