<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Scanner\Probe;

use Magento\Mray\Package\AbstractTree\Component;
use Magento\Mray\Package\AbstractTree\Identifier\Named;
use Magento\Mray\Package\AbstractTree\Identifier\MagentoTheme as IdentifierMagentoTheme;
use Magento\Mray\Package\AbstractTree\Identifier\Unidentifiable;
use Magento\Mray\Package\AbstractTree\Node\Identifier;
use Magento\Mray\Index\Scanner\FailedAssumption;
use Magento\Mray\Index\Scanner\Probe;
use Magento\Mray\Index\Scanner\Sample;
use Magento\Mray\Package\AbstractTree\ScannerSubject;
use Throwable;
use function array_filter;
use function array_shift;
use function array_values;
use function count;
use function explode;
use function in_array;
use function sprintf;

class MagentoTheme implements Probe
{
    private const THEME_XML_FILE = 'theme.xml';
    private const REGISTRATION_FILE = 'registration.php';

    /** @var SubjectReader\SimpleXml  */
    private $xmlReader;
    /** @var SubjectReader\MagentoComponentRegistration  */
    private $regReader;

    /**
     * MagentoTheme constructor.
     */
    public function __construct()
    {
        $this->xmlReader = new SubjectReader\SimpleXml();
        $this->regReader = new SubjectReader\MagentoComponentRegistration();
    }

    /**
     * @inheritDoc
     */
    public function check(ScannerSubject $subject): ?Sample
    {
        if (!$subject->contains(self::THEME_XML_FILE) || !$subject->contains(self::REGISTRATION_FILE)) {
            return null;
        }

        try {
            $xml = $subject->read(self::THEME_XML_FILE, $this->xmlReader);
            $regs = array_values(array_filter(
                $subject->read(self::REGISTRATION_FILE, $this->regReader),
                function (array $reg) {
                    return
                        $reg[SubjectReader\MagentoComponentRegistration::TYPE] ===
                        SubjectReader\MagentoComponentRegistration::TYPE_THEME;
                }
            ));
        } catch (Throwable $e) {
            throw new FailedAssumption(
                sprintf('Invalid %s file at %s: %s', self::THEME_XML_FILE, $subject->location(), $e->getMessage()),
                $e->getCode(),
                $e
            );
        }

        if ($xml->getName() !== 'theme') {
            throw new FailedAssumption(sprintf(
                'Unable to read valid module declaration from %s at %s.',
                self::THEME_XML_FILE,
                $subject->location()
            ));
        }
        if (count($regs) > 1) {
            throw new FailedAssumption(sprintf(
                'Exact one declaration of Magento theme is expected in %s at %s.',
                self::REGISTRATION_FILE,
                $subject->location()
            ));
        }

        $identifier = $this->readIdentifier($regs[0][SubjectReader\MagentoComponentRegistration::NAME] ?? null);
        if (!isset($identifier)) {
            $identifier = new Unidentifiable();
        }

        $parent = $this->readIdentifier($xml->theme->parent ?? null, $identifier->area ?? null);

        return new Sample(
            new Component\MagentoTheme($identifier, $parent),
            Sample::NESTED_PACKAGES_NONE
        );
    }

    /**
     * @param string|null $name
     * @param mixed $defaultArea
     * @return Identifier|null
     */
    private function readIdentifier(?string $name, $defaultArea = null): ?Identifier
    {
        $name = (string)$name;
        if (!$name) {
            return null;
        }

        $parts = explode('/', $name, 3);
        if (in_array($parts[0], ['frontend', 'adminhtml', 'doc'])) {
            $area = array_shift($parts);
        } else {
            $area = $defaultArea;
        }

        if (count($parts) !== 2) {
            return new Named($name);
        } else {
            list($vendor, $name) = $parts;
            return new IdentifierMagentoTheme($name, $vendor, $area);
        }
    }
}
