<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Scanner\Probe\SubjectReader;

use Magento\Mray\Package\AbstractTree\ScannerSubjectReader;
use PhpParser\Error;
use PhpParser\Node;
use PhpParser\NodeTraverser;
use PhpParser\NodeVisitor\FindingVisitor;
use PhpParser\NodeVisitor\NameResolver;
use PhpParser\ParserFactory;
use RuntimeException;
use function dirname;
use function file_get_contents;
use function sprintf;
use function strtolower;

class MagentoComponentRegistration implements ScannerSubjectReader
{
    public const TYPE = 'type';
    public const NAME = 'name';
    public const LOCATION = 'location';

    public const TYPE_MODULE = 'module';
    public const TYPE_THEME = 'theme';
    public const TYPE_LANGUAGE = 'language';
    public const TYPE_LIBRARY = 'library';
    public const TYPE_SETUP = 'setup';

    /**
     * @param string $path
     * @return array
     */
    public function read(string $path)
    {
        // phpcs:ignore Generic.PHP.NoSilencedErrors
        $data = @file_get_contents($path);
        if (!$data) {
            throw new RuntimeException(sprintf('Path %s is not valid path to readable file.', $path));
        }

        try {
            $registrations = [];
            foreach ($this->findRegistrationAstNodes($data) as $registration) {
                $registrations[] =  [
                    self::TYPE => $this->mapType($registration->args[0]->value ?? null),
                    self::NAME => $this->evaluateMagentoComponentName($registration->args[1]->value ?? null),
                    self::LOCATION => $this->evaluateLocation($path, $registration->args[2]->value ?? null),
                ];
            }
            return $registrations;
        } catch (Error $e) {
            throw new RuntimeException(
                sprintf('Unable to parser %s as Magento registration file: %s', $path, $e->getMessage()),
                $e->getCode(),
                $e->getMessage()
            );
        }
    }

    /**
     * @param string $data
     * @return array
     */
    private function findRegistrationAstNodes(string $data): array
    {
        $parser = (new ParserFactory())->create(ParserFactory::ONLY_PHP7);
        $traverser = new NodeTraverser();
        $traverser->addVisitor(new NameResolver());
        $registrationFinder = new FindingVisitor(function (Node $node) {
            if (!$node instanceof Node\Expr\StaticCall) {
                return false;
            }
            // phpcs:ignore Magento2.PHP.LiteralNamespaces
            if ((string)$node->class !== 'Magento\Framework\Component\ComponentRegistrar') {
                return false;
            }
            if ((string)$node->name !== 'register') {
                return false;
            }
            return true;
        });
        $traverser->addVisitor($registrationFinder);
        $traverser->traverse($parser->parse($data));

        return $registrationFinder->getFoundNodes();
    }

    /**
     * @param Node|null $node
     * @return string|null
     */
    private function mapType(?Node $node): ?string
    {
        $type = null;
        if ($node instanceof Node\Expr\ClassConstFetch) {
            $type = $this->mapTypeConst($node);
        } elseif ($node instanceof Node\Scalar\String_) {
            $type = $node->value;
        }

        switch (strtolower((string)$type)) {
            case 'module':
                return self::TYPE_MODULE;
            case 'theme':
                return self::TYPE_THEME;
            case 'language':
                return self::TYPE_LANGUAGE;
            case 'library':
                return self::TYPE_LIBRARY;
            case 'setup':
                return self::TYPE_SETUP;
            default:
                return null;
        }
    }

    /**
     * @param Node\Expr\ClassConstFetch $node
     * @return string|null
     */
    private function mapTypeConst(Node\Expr\ClassConstFetch $node): ?string
    {
        // phpcs:ignore Magento2.PHP.LiteralNamespaces
        if ((string)$node->class !== 'Magento\Framework\Component\ComponentRegistrar') {
            return null;
        }

        return strtolower((string)$node->name);
    }

    /**
     * @param Node $identifier
     * @return string|null
     */
    private function evaluateMagentoComponentName(Node $identifier): ?string
    {
        if ($identifier instanceof Node\Scalar\String_) {
            return (string)$identifier->value;
        }
        return null;
    }

    /**
     * @param string $sourcePath
     * @param Node $location
     * @return string|null
     */
    private function evaluateLocation(string $sourcePath, Node $location): ?string
    {
        if ($location instanceof Node\Scalar\MagicConst\Dir) {
            return dirname($sourcePath);
        }

        if ($location instanceof Node\Expr\FuncCall &&
            (string)$location->name === 'dirname' &&
            isset($location->args[0]) &&
            $location->args[0]->value instanceof Node\Scalar\MagicConst\File
        ) {
            return dirname($sourcePath);
        }

        return null;
    }
}
