<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Scanner\Subject;

use Magento\Mray\Package\AbstractTree\ScannerSubject;
use Magento\Mray\Package\AbstractTree\ScannerSubjectReader;
use RuntimeException;
use function file_exists;
use function get_class;
use function in_array;
use function sprintf;
use function strtolower;
use function substr;
use const DIRECTORY_SEPARATOR;

class Local implements ScannerSubject
{
    /** @var string  */
    private $path;

    /** @var array  */
    private $misses = [];
    /** @var array  */
    private $reads = [];

    /**
     * @param string $path
     */
    public function __construct(string $path)
    {
        $this->path = $path;
    }

    /**
     * @return string
     */
    public function location(): string
    {
        return $this->path;
    }

    /**
     * @param string $relativePath
     * @return bool
     */
    public function contains(string $relativePath): bool
    {
        if (!isset($this->misses[$relativePath])) {
            // phpcs:ignore Generic.PHP.NoSilencedErrors
            $this->misses[$relativePath] = !(@file_exists($this->buildFullPath($relativePath)));
        }
        return !$this->misses[$relativePath];
    }

    /**
     * @param string $relativePath
     * @param ScannerSubjectReader $reader
     * @return mixed
     */
    public function read(string $relativePath, ScannerSubjectReader $reader)
    {
        if (isset($this->misses[$relativePath]) && $this->misses[$relativePath]) {
            throw new RuntimeException(sprintf('Requested file "%s" does not exists.', $relativePath));
        }

        $fileInterpretation = get_class($reader);
        if (isset($this->reads[$fileInterpretation][$relativePath])) {
            return $this->reads[$fileInterpretation][$relativePath];
        }

        $fullPath = $this->buildFullPath($relativePath);
        // phpcs:ignore Generic.PHP.NoSilencedErrors
        if (!@file_exists($fullPath)) {
            $this->misses[$relativePath] = true;
            throw new RuntimeException(sprintf('Requested file "%s" does not exists.', $relativePath));
        }

        $interpretation = $reader->read($fullPath);
        $this->reads[$fileInterpretation][$relativePath] = $interpretation;
        return $this->reads[$fileInterpretation][$relativePath];
    }

    /**
     * @inheritDoc
     */
    public function fragments(): iterable
    {
        if (!is_dir($this->path)) {
            yield from [];
            return;
        }

        foreach (scandir($this->path) as $childPath) {
            if (substr($childPath, 0, 1) === '.') {
                continue;
            }
            if (in_array(strtolower($childPath), ['test', 'tests'])) {
                continue;
            }
            yield new static($this->buildFullPath($childPath));
        }
    }

    /**
     * @param string $relativePath
     * @return string
     */
    private function buildFullPath(string $relativePath): string
    {
        return $this->path . DIRECTORY_SEPARATOR . $relativePath;
    }
}
