<?php
/**
 * Copyright 2022 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Scanner\Tuner\Filter;

use Magento\Mray\Index\Scanner\Probe\SubjectReader\Json;
use Magento\Mray\Package\AbstractTree\Component\MagentoEcosystemAsset;
use Magento\Mray\Package\AbstractTree\DistributionPackage\Composer;
use Magento\Mray\Package\AbstractTree\Identifier\ComposerPackage;
use Magento\Mray\Package\AbstractTree\Identifier\Versioned;
use Magento\Mray\Package\AbstractTree\Node\Package;

class MagentoAssetsAndExtendedLibs implements FilterInterface
{
    private const COMPOSER_LOCK = 'composer.lock';
    private const PRODUCT_PACKAGE = 'magento/product-enterprise-edition';
    private const SKIP_PACKAGES = [
        'zendframework/zend-validator' => true
    ];

    /**
     * @var string
     */
    private $projectRootPath;

    /**
     * @var Json
     */
    private $jsonReader;

    /**
     * @param string $projectRootPath
     */
    public function __construct(string $projectRootPath)
    {
        $this->projectRootPath = $projectRootPath;
    }

    /**
     * @param Package[] $packages
     * @return Package[]
     */
    public function filter(array $packages): array
    {
        $dependencies = $this->getDirectDependencies();
        return array_filter($packages, function (Package $package) use ($dependencies) {
            if ($package instanceof MagentoEcosystemAsset) {
                return true;
            }
            if (!$package instanceof Composer) {
                return false;
            }
            if (!$package instanceof Composer) {
                return false;
            }
            $identifier = ($package->getIdentifier() instanceof Versioned)
                ? $package->getIdentifier()->getIdentifier()
                : $package->getIdentifier();
            if ($identifier instanceof ComposerPackage
                && isset($dependencies[(string) $identifier])
                && !isset(self::SKIP_PACKAGES[(string) $identifier])
            ) {
                return true;
            }
            return false;
        });
    }

    /**
     * @return array
     */
    private function getDirectDependencies(): array
    {
        $composerLockPath = rtrim($this->projectRootPath, '/') . '/' . self::COMPOSER_LOCK;
        if (!file_exists($composerLockPath)) {
            throw new \InvalidArgumentException(sprintf('File does not exist: %s', $composerLockPath));
        }

        $data = $this->getJsonReader()->read($composerLockPath);

        if (!isset($data['packages']) || !is_array($data['packages']) || empty($data['packages'])) {
            throw new \RuntimeException(sprintf('Cannot find "require" section in %s.', $composerLockPath));
        }

        foreach ($data['packages'] as $library) {
            if (isset($library['name']) && $library['name'] === self::PRODUCT_PACKAGE) {
                return $this->getRequiredPackages($library);
            }
        }

        throw new \RuntimeException(
            sprintf('Cannot find "%s" package in %s.', self::PRODUCT_PACKAGE, $composerLockPath)
        );
    }

    /**
     * @param array $library
     * @return array
     */
    private function getRequiredPackages(array $library): array
    {
        if (!isset($library['require']) || !is_array($library['require']) || empty($library['require'])) {
            throw new \RuntimeException(
                sprintf(
                    'Cannot find "require" section for "%s".',
                    self::PRODUCT_PACKAGE
                )
            );
        }

        $thirdPartyDependencies = [];
        foreach ($library['require'] as $dependency => $version) {
            if (strpos($dependency, 'magento') === 0 || strpos($dependency, 'ext') === 0) {
                continue;
            }
            $thirdPartyDependencies[$dependency] = true;
        }
        return $thirdPartyDependencies;
    }

    /**
     * @return Json
     */
    private function getJsonReader(): Json
    {
        if (!$this->jsonReader) {
            $this->jsonReader = new Json();
        }
        return $this->jsonReader;
    }
}
