<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Scanner\Tuner;

use Magento\Mray\CodeStructuralElement\Php\NodeVisitor\MagentoApiPropagation;
use Magento\Mray\CodeStructuralElement\Php\NodeVisitor\PhpDocAst;
use Magento\Mray\Package\AbstractTree\Node\PhpCode;
use Magento\Mray\Package\AbstractTree\Node\PhpFile;
use Magento\Mray\Index\Scanner\Tuner;
use Magento\Mray\Package\AbstractTree\ScannerSubject;
use PhpParser\ErrorHandler;
use PhpParser\Lexer\Emulative as EmulativeLexer;
use PhpParser\NodeTraverser;
use PhpParser\NodeVisitor\NameResolver;
use PhpParser\NodeVisitor\ParentConnectingVisitor;
use PhpParser\Parser;
use PhpParser\ParserFactory;
use function basename;
use function file_get_contents;
use function iterator_to_array;
use function strcasecmp;
use function strlen;
use function substr;

class MagentoPhpCodeNodeSetter implements Tuner
{
    public const NODE_PHP_CODE = 'phpCode';

    public const IGNORE_REGISTRATION_FILE = 1;
    public const IGNORE_VENDOR = 2;
    public const IGNORE_GENERATED = 4;

    /**
     * @var int
     */
    private $options = self::IGNORE_REGISTRATION_FILE | self::IGNORE_VENDOR | self::IGNORE_GENERATED;

    /**
     * @var Parser
     */
    private $phpParser;

    /**
     * @var ErrorHandler\Collecting
     */
    private $errorHandler;

    /**
     * @var NodeTraverser
     */
    private $traverser;

    /**
     * @var Filter\FilterInterface
     */
    private $packagesFilter;

    /**
     * @param Filter\FilterInterface $packagesFilter
     */
    public function __construct(Tuner\Filter\FilterInterface $packagesFilter)
    {
        $this->packagesFilter = $packagesFilter;
    }

    /**
     * @inheritDoc
     */
    public function tune(ScannerSubject $subject, array $packages): array
    {
        $filteredPackages = $this->packagesFilter->filter($packages);

        if (empty($filteredPackages)) {
            return $packages;
        }

        $phpFiles = iterator_to_array(
            $this->findPhpCode($subject, strlen($subject->location()) + 1),
            false
        );

        if (!$phpFiles) {
            return $packages;
        }

        foreach ($filteredPackages as $package) {
            $package->{self::NODE_PHP_CODE} = new PhpCode($phpFiles);
        }

        return $packages;
    }

    /**
     * @param ScannerSubject $subject
     * @param int $sharedLocation
     * @return iterable
     */
    private function findPhpCode(ScannerSubject $subject, int $sharedLocation): iterable
    {
        foreach ($subject->fragments() as $fragment) {
            $bn = basename($fragment->location());
            if (($this->options & self::IGNORE_VENDOR) && $bn === 'vendor') {
                continue;
            }
            if (($this->options & self::IGNORE_GENERATED) && $bn === 'generated') {
                continue;
            }
            if (($this->options & self::IGNORE_REGISTRATION_FILE) && $bn === 'registration.php') {
                continue;
            }

            if (strcasecmp(pathinfo($fragment->location(), PATHINFO_EXTENSION), 'php') === 0 ||
                strcasecmp(pathinfo($fragment->location(), PATHINFO_EXTENSION), 'phtml') === 0) {
                // phpcs:ignore Generic.PHP.NoSilencedErrors
                $code = @file_get_contents($fragment->location());
                if (!$code) {
                    continue;
                }

                $traversedNodes = $this->getTraverser()->traverse(
                    $this->getPhpParser()->parse($code, $this->errorHandler) ?? []
                );
                $errors = $this->getErrorHandler()->getErrors();
                $this->errorHandler->clearErrors();

                yield new PhpFile(
                    substr($fragment->location(), $sharedLocation),
                    $traversedNodes,
                    $errors
                );
            } elseif (!$fragment->contains('composer.json') &&
                !(
                    $fragment->contains('etc/module.xml') ||
                    $fragment->contains('theme.xml') ||
                    $fragment->contains('language.xml')
                )
            ) {
                yield from $this->findPhpCode($fragment, $sharedLocation);
            }
        }
        return;
        // phpcs:ignore Squiz.PHP.NonExecutableCode
        yield;
    }

    /**
     * @return Parser
     */
    private function getPhpParser(): Parser
    {
        if (!$this->phpParser) {
            $this->phpParser = (new ParserFactory())->create(
                ParserFactory::ONLY_PHP7,
                new EmulativeLexer([
                    'phpVersion' => '7.4',
                    'usedAttributes' => [
                        'comments', // capture comments
                        'startLine', // capture node start line in file
                        'endLine', // capture node end line in file
                        'startFilePos', // capture position in file to convert to column in start line
                        'endFilePos' // capture position in file to convert to column in end line
                    ],
                ])
            );
        }
        return $this->phpParser;
    }

    /**
     * @return NodeTraverser
     */
    private function getTraverser(): NodeTraverser
    {
        if (!$this->traverser) {
            $this->traverser = new NodeTraverser();
            $this->traverser->addVisitor(new ParentConnectingVisitor());
            $this->traverser->addVisitor(new NameResolver($this->getErrorHandler()));
            $this->traverser->addVisitor(new PhpDocAst());
            $this->traverser->addVisitor(new MagentoApiPropagation());
        }
        return $this->traverser;
    }

    /**
     * @return ErrorHandler\Collecting
     */
    private function getErrorHandler(): ErrorHandler\Collecting
    {
        if (!$this->errorHandler) {
            $this->errorHandler = new ErrorHandler\Collecting();
        }
        return $this->errorHandler;
    }
}
