<?php
/**
 * Copyright 2022 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Traverser;

use Magento\Mray\Index\UseCasesFilter;
use Magento\Mray\Package\AbstractTree\Component\MagentoModule;
use Magento\Mray\Package\AbstractTree\Component\MagentoProject;
use Magento\Mray\Package\AbstractTree\Node\Node;
use Magento\Mray\Parse\Xml\GetDbSchemaUsageCases;

class DbSchemaUsageDiscovery implements NodeVisitor
{
    /**
     * @var string
     */
    private $version;

    /**
     * @var GetDbSchemaUsageCases
     */
    private $getUsageCases;

    /**
     * @param string $version
     */
    public function __construct(string $version)
    {
        $this->version = $version;
        $this->getUsageCases = new GetDbSchemaUsageCases();
    }

    /**
     * @inheritDoc
     */
    public function leaveNode(Node $node)
    {
        if (!$node instanceof MagentoModule && !$node instanceof MagentoProject) {
            return null;
        }

        if (!isset($node->attributes['dbSchema']) || !is_array($node->attributes['dbSchema'])) {
            return null;
        }

        $dependencies = $node->getAttribute('dependencies') ?? [];

        foreach ($node->attributes['dbSchema'] as $path => $content) {
            $dbSchemaUsageCases = $this->getUsageCases->execute($content, $path, $this->version);
            foreach ($dbSchemaUsageCases as $what => $usageCase) {
                $dependencies[$what] = [
                    'type' => 'db',
                    'realName' => $usageCase->getWhat(),
                    'usage' => [
                        $usageCase->getHow() => [
                            $usageCase->describe()
                        ]
                    ]
                ];
            }
        }

        $node->setAttribute('dependencies', $dependencies);
    }

    /**
     * @param  Node $node
     * @return null
     */
    public function enterNode(Node $node)
    {
        return null;
    }
}
