<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Traverser;

use Magento\Mray\CodeStructuralElement\Php\NodeRuntimeTypeResolver\Factory as TypeResolverFactory;
use Magento\Mray\CodeStructuralElement\Php\NodeVisitor\ContextSwitcher;
use Magento\Mray\CodeStructuralElement\Php\NodeVisitor\ExpressionRuntimeTypeResolver;
use Magento\Mray\CodeStructuralElement\Php\NodeVisitor\VariableDeclarationReflector;
use Magento\Mray\CodeStructuralElement\Php\Reflection\Context;
use Magento\Mray\Package\AbstractTree\Node\Node;
use Magento\Mray\Package\AbstractTree\Node\Package;
use Magento\Mray\Package\AbstractTree\Node\PhpCode;


use PhpParser\NodeTraverser;
use function array_filter;
use function get_object_vars;

class PhpCodeRuntimeTypesEstimation implements NodeVisitor
{
    /** @var Context\StructuralElementDeclarations|null  */
    private $declarations;

    /**
     * @param Context\StructuralElementDeclarations|null $declarations
     */
    public function __construct(?Context\StructuralElementDeclarations $declarations = null)
    {
        $this->declarations = $declarations;
    }

    /**
     * @inheritDoc
     */
    public function enterNode(Node $node)
    {
        if (!$node instanceof Package) {
            return null;
        }

        foreach ($this->getPhpCode($node) as $phpCode) {
            foreach ($phpCode->files as $f) {
                $context = new Context\Pointer(new Context\Global_($this->declarations));
                $traverser = new NodeTraverser();
                $traverser->addVisitor(new ContextSwitcher($context));
                $traverser->addVisitor(new VariableDeclarationReflector($context));
                $traverser->addVisitor(new ExpressionRuntimeTypeResolver(
                    TypeResolverFactory::default()->create(),
                    $context
                ));
                $traverser->traverse($f->ast);
            }
        }

        return null;
    }

    /**
     * @param Package $pkg
     * @return PhpCode[]
     */
    private function getPhpCode(Package $pkg): array
    {
        return array_filter(get_object_vars($pkg), function ($node) {
            return $node instanceof PhpCode;
        });
    }

    /**
     * @param Node $node
     * @return null
     */
    public function leaveNode(Node $node)
    {
        return null;
    }
}
