<?php
/**
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Index\Traverser;

use Magento\Mray\CodeStructuralElement\Php\NodeVisitor\UsageDiscovery;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassConstantDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\ClassLikeMemberDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\Context;
use Magento\Mray\CodeStructuralElement\Php\Reflection\FunctionDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\GlobalConstantDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\InCodeDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\InterfaceDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\MethodDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\PropertyDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\StructuralElementDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Reflection\TraitDeclaration;
use Magento\Mray\CodeStructuralElement\Php\Usage\UsageCase;
use Magento\Mray\CodeStructuralElement\Php\Usage\UsageRegistry;
use Magento\Mray\MagentoApiIndex\Index\Version;
use Magento\Mray\Package\AbstractTree\Node\Node;
use Magento\Mray\Package\AbstractTree\Node\Package;
use Magento\Mray\Package\AbstractTree\Node\PhpCode;
use Magento\Mray\Package\AbstractTree\Node\PhpFile;
use PhpParser\NodeTraverser;
use function array_filter;
use function array_merge;
use function explode;
use function get_object_vars;
use function ltrim;
use function strlen;
use function strtolower;
use function strtoupper;
use function substr;
use const DIRECTORY_SEPARATOR;

class PhpCodeUsageDiscovery implements NodeVisitor
{
    /**
     * @var Context\StructuralElementDeclarations
     */
    private $declarations;

    /**
     * @var PhpCodeUsageFilterInterface
     */
    private $usageFilter;

    /**
     * @var Version
     */
    private $index;

    /**
     * @param Context\StructuralElementDeclarations $declarations
     * @param PhpCodeUsageFilterInterface $codeUsageFilter
     * @param Version $index
     */
    public function __construct(
        Context\StructuralElementDeclarations $declarations,
        PhpCodeUsageFilterInterface $codeUsageFilter,
        Version $index
    ) {
        $this->declarations = $declarations;
        $this->usageFilter = $codeUsageFilter;
        $this->index = $index;
    }

    /**
     * @param Node $node
     * @return null
     */
    public function enterNode(Node $node)
    {
        return null;
    }

    /**
     * @param Node $node
     * @return null
     */
    public function leaveNode(Node $node)
    {
        if (!$node instanceof Package) {
            return null;
        }

        $usageRegistry = new UsageRegistry();
        $internals = [];
        foreach ($this->getPhpCode($node) as $phpCode) {
            foreach ($phpCode->files as $phpFile) {
                if (!$phpFile->getDeclarations()) {
                    continue;
                }

                foreach ($phpFile->getDeclarations() as $fqsen => $declarations) {
                    foreach ($declarations as $declaration) {
                        if (!$declaration instanceof InCodeDeclaration) {
                            return null;
                        }

                        $internals[$declaration->getName()] = true;

                        $ast = [$declaration->getDeclarationNodes()[0]];
                        $traverse = new NodeTraverser();
                        $traverse->addVisitor(new UsageDiscovery(
                            $usageRegistry,
                            $this->getLocationInformation($node, $phpFile)
                        ));
                        $traverse->traverse($ast);
                    }
                }
            }
        }

        $dependencies = [];
        foreach ($this->getFilteredUsageCases($usageRegistry, $internals) as $what => $usageGroup) {
            $whatData = $this->getDataByName($what);
            // phpcs:ignore Magento2.Performance.ForeachArrayMerge
            $dependencies[$what] = array_merge(
                $dependencies[$what] ?? [],
                $whatData
            );
            foreach ($usageGroup as $how => $usageCases) {
                $how = explode('-', $how, 2);
                $type = $how[1] ?? null;
                $how = $how[0];
                foreach ($usageCases as $usageCase) {
                    if ($type && $dependencies[$what]['type'] === 'unknown') {
                        $dependencies[$what]['type'] = $type;
                    }

                    $dependencies[$what]['usage'][$how][] = $usageCase->describe();
                }
            }
        }

        $node->setAttribute('dependencies', $dependencies);

        return null;
    }

    /**
     * @param UsageRegistry $usageRegistry
     * @param array $internals
     * @return UsageCase[][][]
     */
    private function getFilteredUsageCases(UsageRegistry $usageRegistry, array $internals): array
    {
        $filteredUsages = [];
        foreach ($usageRegistry->getUsages() as $what => $usageGroup) {
            if (isset($internals[explode('::', $what)[0]]) ||
                !$this->usageFilter->filter($what, $this->index->magentoVersion())) {
                continue;
            }
            foreach ($usageGroup as $how => $usageCases) {
                if ($how === 'extend-method' && $this->index->isApi($what) === null) {
                    continue;
                }
                $filteredUsages[$what][$how] = $usageCases;
            }
        }
        return $filteredUsages;
    }

    /**
     * @param Package $pkg
     * @param PhpFile $file
     * @return string[]
     */
    private function getLocationInformation(Package $pkg, PhpFile $file)
    {
        return [
            'file' => $pkg->getAttribute('path') . DIRECTORY_SEPARATOR . $file->file,
        ];
    }

    /**
     * @param Package $pkg
     * @return PhpCode[]
     */
    private function getPhpCode(Package $pkg): array
    {
        return array_filter(get_object_vars($pkg), function ($node) {
            return $node instanceof PhpCode;
        });
    }

    /**
     * @param string $fqsen
     * @return array|string[]
     */
    private function getDataByName(string $fqsen): array
    {
        $fqsen = explode('::', $fqsen);
        $topLevelName = $fqsen[0];
        $memberName = $fqsen[1] ?? null;
        if (!$this->declarations) {
            return $this->estimateInformationFromName($topLevelName, $memberName);
        }

        $ds = $this->declarations->findClassLike($topLevelName) ?:
              $this->declarations->findFunction($topLevelName) ?:
              $this->declarations->findGlobalConstant($topLevelName);

        if (empty($ds)) {
            return $this->estimateInformationFromName($topLevelName, $memberName);
        }

        if (!$memberName) {
            return $this->getInformationFromDeclaration($ds[0]);
        }

        $clearMemberName = ltrim($memberName, '$');
        foreach ($ds as $declaration) {
            if (!$declaration instanceof ClassLikeDeclaration) {
                continue;
            }
            foreach ($declaration->getMembers() as $md) {
                if ($md->getName() !== $clearMemberName) {
                    continue;
                }
                if ($clearMemberName !== $memberName && !$md instanceof PropertyDeclaration) {
                    continue;
                }
                if ($clearMemberName === $memberName && $md instanceof PropertyDeclaration) {
                    continue;
                }
                return $this->getInformationFromDeclaration($md);
            }
        }
        return $this->estimateInformationFromName($topLevelName, $memberName);
    }

    /**
     * @param StructuralElementDeclaration $d
     * @return array
     */
    private function getInformationFromDeclaration(StructuralElementDeclaration $d)
    {
        if ($d instanceof InterfaceDeclaration) {
            $type = 'interface';
        } elseif ($d instanceof ClassDeclaration) {
            $type = 'class';
        } elseif ($d instanceof TraitDeclaration) {
            $type = 'trait';
        } elseif ($d instanceof FunctionDeclaration) {
            $type = 'function';
        } elseif ($d instanceof GlobalConstantDeclaration) {
            $type = 'constant';
        } elseif ($d instanceof MethodDeclaration) {
            $type = 'method';
        } elseif ($d instanceof PropertyDeclaration) {
            $type = 'property';
        } elseif ($d instanceof ClassConstantDeclaration) {
            $type = 'constant';
        } else {
            $type = 'unknown';
        }

        $realName = $d->getName();
        if ($d instanceof ClassLikeMemberDeclaration) {
            if ($d instanceof PropertyDeclaration) {
                $realName = '$' . $realName;
            }
            $realName = $d->getClassLikeName() . '::' . $realName;
        }

        $info = [
            'type' => $type,
            'realName' => $realName,
        ];
        if (isset($d->api)) {
            $info['api'] = $d->api;
        }
        if (isset($d->deprecated)) {
            $info['deprecated'] = $d->deprecated;
        }
        return $info;
    }

    /**
     * @param string $topLevelName
     * @param string|null $memberName
     * @return string[]
     */
    private function estimateInformationFromName(string $topLevelName, ?string $memberName)
    {
        if (isset($memberName)) {
            if (substr($memberName, 0, 1) === '$') {
                return ['type' => 'property'];
            } elseif (strtoupper($memberName) === $memberName) {
                return ['type' => 'constant'];
            } else {
                return ['type' => 'method'];
            }
        }
        foreach (['interface', 'able'] as $marker) {
            if (strtolower(substr($topLevelName, -strlen($marker))) === $marker) {
                return ['type' => 'interface'];
            }
        }
        return ['type' => 'class'];
    }
}
