<?php
/**
 * Copyright 2022 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it.
 */
declare(strict_types=1);

namespace Magento\Mray\Parse\Xml\DbSchemaUsageCases;

use Magento\Mray\CodeStructuralElement\Php\Usage\UsageCase;
use Magento\Mray\MagentoApiIndex\Api\DbSchemaInfo;
use SimpleXMLElement;

class DbSchemaTable
{
    private const XPATH_FOR_ELEMENTS = [
        'table' => '//table[@name="%s"]',
        'column' => '//table[@name="%s"]/column[@name="%s"]',
        'constraint' => '//table[@name="%s"]/constraint[@referenceId="%s"]',
        'index' => '//table[@name="%s"]/index[@referenceId="%s"]',
    ];

    /**
     * @var DbSchemaInfo
     */
    private $dbSchemaInfo;

    /**
     * @var string
     */
    private $tableName;

    /**
     * @var array
     */
    private $tableData;

    /**
     * @var SimpleXMLElement
     */
    private $fileXml;

    /**
     * @var string
     */
    private $filePath;

    /**
     * @var string
     */
    private $version;

    /**
     * @param string $tableName
     * @param array $tableData
     * @param SimpleXMLElement $fileXml
     * @param string $filePath
     * @param string $version
     */
    public function __construct(
        string $tableName,
        array $tableData,
        SimpleXMLElement $fileXml,
        string $filePath,
        string $version
    ) {
        $this->tableName = $tableName;
        $this->tableData = $tableData;
        $this->fileXml = $fileXml;
        $this->filePath = $filePath;
        $this->version = $version;
        $this->dbSchemaInfo = new DbSchemaInfo();
    }

    /**
     * @return UsageCase[]
     */
    public function getUsageCases(): array
    {
        if (isset($this->tableData['constraint'])) {
            foreach ($this->tableData['constraint'] as $constraintName => $constraintData) {
                if (isset($constraintData['referenceTable'])
                    && $this->dbSchemaInfo->getTable($constraintData['referenceTable'], $this->version) !== null
                ) {
                    $usageCases[$this->tableName . '.' . $constraintName] = $this->crateUsageCase(
                        'constraint',
                        'added-constraint-referencing-core-table',
                        $constraintData,
                        $constraintName
                    );
                }
            }
        }

        if ($this->dbSchemaInfo->getTable($this->tableName, $this->version) === null) {
            $usageCases[$this->tableName] = $this->crateUsageCase('table', 'added-table', $this->tableData);
            return $usageCases;
        }

        $usageCases[$this->tableName] = $this->crateUsageCase(
            'table',
            'extended-core-table',
            $this->tableData
        );
        if (isset($this->tableData['column'])) {
            foreach ($this->tableData['column'] as $columnName => $columnData) {
                $usageCase = $this->getColumnUsageCase($columnName, $columnData);
                $usageCases[$usageCase->getWhat()] = $usageCase;
            }
        }

        if (isset($this->tableData['index'])) {
            foreach ($this->tableData['index'] as $indexName => $indexData) {
                $usageCase = $this->getIndexUsageCase($indexName, $indexData);
                $usageCases[$usageCase->getWhat()] = $usageCase;
            }
        }

        if (isset($this->tableData['constraint'])) {
            foreach ($this->tableData['constraint'] as $constraintName => $constraintData) {
                $usageCase = $this->getCoreConstraintUsageCase($constraintName, $constraintData);
                $usageCases[$usageCase->getWhat()] = $usageCase;
            }
        }

        return $usageCases;
    }

    /**
     * @param string $name
     * @param array $data
     * @return UsageCase
     */
    private function getIndexUsageCase(string $name, array $data): UsageCase
    {
        $isCoreIndex = $this->dbSchemaInfo->getIndex($this->tableName, $name, $this->version) !== null;

        return $this->crateUsageCase(
            'index',
            $isCoreIndex ? 'extended-core-table-index' : 'added-index-to-core-table',
            $data,
            $name
        );
    }

    /**
     * @param string $name
     * @param array $data
     * @return UsageCase
     */
    private function getColumnUsageCase(string $name, array $data): UsageCase
    {
        $isCoreColumn = $this->dbSchemaInfo->getColumn(
            $this->tableName,
            $name,
            $this->version
        ) !== null;

        return $this->crateUsageCase(
            'column',
            $isCoreColumn ? 'extended-core-table-column' : 'added-column-to-core-table',
            $data,
            $name
        );
    }

    /**
     * @param string $name
     * @param array $data
     * @return UsageCase
     */
    private function getCoreConstraintUsageCase(string $name, array $data): UsageCase
    {
        $isCoreConstraint = $this->dbSchemaInfo->getConstraint($this->tableName, $name, $this->version) !== null;

        return $this->crateUsageCase(
            'constraint',
            $isCoreConstraint ? 'extended-core-table-constraint' : 'added-constraint-to-core-table',
            $data,
            $name
        );
    }

    /**
     * @param string $dependencyType
     * @param string $usageType
     * @param array $details
     * @param string|null $childName
     * @return UsageCase
     */
    private function crateUsageCase(
        string $dependencyType,
        string $usageType,
        array $details,
        string $childName = null
    ): UsageCase {
        $lineNumber = $this->getLineNumber($dependencyType, $childName);
        return new UsageCase(
            $childName ? $this->tableName . '.' . $childName : $this->tableName,
            $usageType,
            [
                'file' => $this->filePath,
                'position' => [
                    'startLine' => $lineNumber,
                    'endLine' => $lineNumber
                ]
            ],
            $details
        );
    }

    /**
     * @param string $type
     * @param string|null $childName
     * @return int
     */
    private function getLineNumber(
        string $type,
        ?string $childName = null
    ): int {
        $xpath = $childName ? sprintf(self::XPATH_FOR_ELEMENTS[$type], $this->tableName, $childName)
            : sprintf(self::XPATH_FOR_ELEMENTS[$type], $this->tableName);

        $element = $this->fileXml->xpath($xpath);

        if (!$element || !is_array($element)) {
            return 1;
        }

        $domElement = dom_import_simplexml(current($element));

        return $domElement ? $domElement->getLineNo() : 1;
    }
}
