<?php
declare(strict_types=1);
namespace Mtc\AdminShippingMethod\Model;

use Magento\Framework\App\RequestInterface;
use Magento\Quote\Model\Quote\Address\RateRequest;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Quote\Model\Quote\Address\RateResult\ErrorFactory;
use Mtc\AdminShippingMethod\Helper\Data;
use Psr\Log\LoggerInterface;
use Magento\Shipping\Model\Rate\ResultFactory;
use Magento\Quote\Model\Quote\Address\RateResult\MethodFactory;
use Magento\Framework\App\State;
use Magento\Shipping\Model\Carrier\AbstractCarrier;
use Magento\Shipping\Model\Carrier\CarrierInterface;
use Magento\Backend\App\Area\FrontNameResolver;
use Magento\Framework\Exception\LocalizedException;
use Magento\Shipping\Model\Rate\Result;
use Magento\Quote\Model\Quote\Address\RateResult\Method as RateResultMethod;

/**
 * Courier rate shipping model
 *
 * @api
 */
class ShippingMethod extends AbstractCarrier
{
    /**
     * @var string
     */
    protected $_code = 'adminshippingmethod';
    
    /**
     * @var bool
     */
    protected $_isFixed = true;
    
    /**
     * @var ResultFactory
     */
    protected $rateResultFactory;
    
    /**
     * @var MethodFactory
     */
    protected $rateMethodFactory;
    
    /**
     * @var \Magento\Quote\Model\Quote\Address\RateResult\ErrorFactory
     */
    protected $rateErrorFactory;
    
    /**
     * @var State
     */
    protected $appState;
    
    /**
     * @param \Magento\Framework\App\State                                $appState
     * @param \Magento\Framework\App\RequestInterface                     $requestInterface
     * @param \Magento\Framework\App\Config\ScopeConfigInterface          $scopeConfig
     * @param \Magento\Quote\Model\Quote\Address\RateResult\ErrorFactory  $rateErrorFactory
     * @param \Psr\Log\LoggerInterface                                    $logger
     * @param \Magento\Shipping\Model\Rate\ResultFactory                  $rateResultFactory
     * @param \Magento\Quote\Model\Quote\Address\RateResult\MethodFactory $rateMethodFactory
     * @param array                                                       $data
     */
    public function __construct(
        ScopeConfigInterface $scopeConfig,
        ErrorFactory $rateErrorFactory,
        LoggerInterface $logger,
        ResultFactory $rateResultFactory,
        MethodFactory $rateMethodFactory,
        State $appState,
        RequestInterface $requestInterface,
        Data $help,
        array $data = []
    ) {
        parent::__construct(
            $scopeConfig,
            $rateErrorFactory,
            $logger,
            $data
        );
        $this->rateResultFactory = $rateResultFactory;
        $this->rateMethodFactory = $rateMethodFactory;
        $this->appState = $appState;
        $this->help = $help;
        $this->requestInterface = $requestInterface;
        $this->_scopeConfig = $scopeConfig;
    }

    /**
     * Checks if user is logged in as admin
     *
     * @return bool
     * @throws LocalizedException
     */
    protected function isAdmin(): bool
    {
        if ($this->appState->getAreaCode() === FrontNameResolver::AREA_CODE) {
            return true;
        }
        return false;
    }

    /**
     * @param \Magento\Quote\Model\Quote\Address\RateRequest $request
     *
     * @return false|\Magento\Shipping\Model\Rate\Result
     * @throws \Magento\Framework\Exception\InputException
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function collectRates(RateRequest $request)
    {
        if (!$this->getConfigFlag('active') || !$this->isAdmin()) {
            return false;
        }

        /**
         * @var Result $result
        */
        $result = $this->rateResultFactory->create();
        /**
         * @var Method $method
        */
        $method = $this->rateMethodFactory->create();
        
        $method->setCarrier('adminshippingmethod');
        $method->setCarrierTitle($this->help->getTitle($this->idStore()));

        $method->setMethod('adminshippingmethod');
        $method->setMethodTitle($this->help->getName($this->idStore()));

        $method->setPrice('0.00');
        $method->setCost('0.00');
        $result->append($method);

        return $result;
    }
    
    /**
     * @return int
     */
    public function idStore() :int
    {
        return (int)((isset($this->requestInterface->getPostValue()[ 'store_id' ])) ?
            $this->requestInterface->getPostValue()[ 'store_id' ] : (($this->getParamStore()) ? : 0));
    }
    
    /**
     * @return array
     */
    public function getAllowedMethods(): array
    {
        return ['freeshippingadmin' => $this->getConfigData('name')];
    }
    
    
    /**
     * Retrieve information from carrier configuration
     *
     * @param   string $field
     * @return  false|string
     */
    public function getConfigData($field)
    {
        if (empty($this->_code)) {
            return false;
        }
        $path = 'carriers/' . $this->_code . '/' . $field;
        
        return $this->_scopeConfig->getValue(
            $path,
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE,
            $this->getStore()
        );
    }
}
