<?php

namespace Mtc\Auction\Bid;

use Carbon\Carbon;
use Mtc\Auction\Bid\Rule;

trait BidLogic
{

    /**
     * Check if this bid can be placed
     *
     * @return bool
     * @throws \Exception
     */
    protected function canPlaceBid()
    {
        /*if (config('auction.settings.registration_fee_for_bidders') === true && empty($this->member->registration_fee_paid_at)) {
            throw new \Exception(__('lots.bidding.must_pay_registration_fee'));
        }*/

        if (!$this->lot->isBiddable() /*|| !$this->member->can_bid*/) {
            throw new \Exception(__('auction::lots.bidding.not_biddable_lot'));
        }

        if (!$this->validBidAmount()) {
            $min_bid = Rule::findNextBidStep($this, $this->previous_bid->bid ?? 0);
            throw new \Exception(__('auction::lots.bidding.too_small_error', ['min_bid' => self::setInUsersCurrency($min_bid)]));
        }

        if (config('auction::auction.settings.minimum_bid_value_per_lot') === true) {
            $this->min_bid_value = $this->lot->min_bid_value;

            if ($this->max_bid < $this->lot->min_bid_value) {
                $error_message = __('auction::lots.bidding.under_minimal_value',
                    ['min_value' => static::setInUsersCurrency($this->lot->min_bid_value)]);
                throw new \Exception($error_message);
            }
        }

        if (empty($this->lot->min_bid_value) && config('auction.settings.minimum_bid_value') > 0
            && $this->max_bid
               < config('auction::auction.settings.minimum_bid_value')) {
            $this->bid->min_bid_value = config('auction::auction.settings.minimum_bid_value');
            $error_message            = __('auction::lots.bidding.under_minimal_value',
                ['min_value' => static::setInUsersCurrency(config('auction::auction.settings.minimum_bid_value'))]);
            throw new \Exception($error_message);
        }

        return true;
    }

    /**
     * Helper method for setting error / success values in users currency
     *
     * @param $value
     *
     * @return string
     */
    public static function setInUsersCurrency($value)
    {
        return $_SESSION['currency']['symbol'] . number_format($_SESSION['currency']['ratio'] * $value, 2);
    }

    /**
     * Check if the bid that will be placed is high enough to be considered valid
     *
     * @return bool
     */
    protected function validBidAmount()
    {
        return !empty(Rule::findLowestValidBidStep($this, $this->previous_bid->bid ?? 0));
    }

    /**
     * Place bid
     *
     * @return mixed
     */
    protected function placeBid()
    {
        
        if ($this->shouldTriggerAntiSniping()) {
            $this->lot->triggerAntiSnipeExtend();
        }
        
        if ($this->isBidUnderReservePrice()) {
            return $this->placeUnderReserveBid();
        }
        
        if ($this->isReserveBreachBid()) {
            return $this->placeReserveBreachBid();
        }

        return $this->placeOverReserveBid();
    }

    /**
     * Bid is still under reserve, it is placed with highest acceptable value
     *
     * @return $this
     */
    protected function placeUnderReserveBid()
    {
        $this->bid = Rule::findHighestValidBidStep($this, $this->previous_bid->bid ?? 0);
        $this->setBidValueInCurrency();
        $this->save();
//        $this->tryNotifySuccessfulBid();
        if ($this->previous_bid && !$this->tryingToChangeMyOwnBid()) {
            $this->previous_bid->tryNotifyOutbid();
        }

        return $this;
    }

    /**
     * Bid just went over reserve value, now we start placing lowest bid possible
     *
     * @return $this
     */
    protected function placeReserveBreachBid()
    {
        $this->bid = Rule::findLowestValidBidStep($this, $this->lot->reserve_price, false);
        if ($this->bid === false) {
            $this->bid = $this->max_bid;
        }
        $this->setBidValueInCurrency();
        $this->save();
//        $this->tryNotifySuccessfulBid();
        if ($this->previous_bid && !$this->tryingToChangeMyOwnBid()) {
            $this->previous_bid->tryNotifyOutbid();
        }

        return $this;
    }

    /**
     * Place bid that is higher than reserve price.
     * We check if we need to do bidding war between 2 people, otherwise increase the bid to next level
     *
     * @return $this
     */
    protected function placeOverReserveBid()
    {
        if (!is_null($this->previous_bid) && $this->tryingToChangeMyOwnBid()) {
            return $this->changeMyOwnBid();
        }
        
        if (!is_null($this->previous_bid) && $this->hasBiddingWar()) {
            return $this->doBiddingWar();
        }
        
        if (!is_null($this->previous_bid)) {
            $this->bid = Rule::findLowestValidBidStep($this, $this->previous_bid->bid ?? 0);
        } else {
            $this->bid = Rule::findLowestValidBidStep($this, 0);
        }
        
        $this->setBidValueInCurrency();
        $this->save();
//        $this->tryNotifySuccessfulBid();

        if ($this->previous_bid && !$this->tryingToChangeMyOwnBid()) {
            $this->previous_bid->tryNotifyOutbid();
        }

        return $this;
    }

    /**
     * Check if user is trying to change their max bid
     *
     * @return bool
     */
    protected function tryingToChangeMyOwnBid()
    {
        if (!is_null($this->previous_bid) && !is_null($this->member_id)) {
            return $this->member_id == $this->previous_bid->member_id;
        }

        return false;
    }

    /**
     * Increase the max bid value for my own bid.
     * Instead of placing a new bid we simply update and return the max value of the previous bid
     *
     * @return mixed
     */
    protected function changeMyOwnBid()
    {
        $this->previous_bid->max_bid = $this->max_bid;
        $this->previous_bid->save();

        //$this->previous_bid->tryNotifySuccessfulBid();
        return $this->previous_bid;
    }

    /**
     * Check if the previous bidder has ability to increase their bid value
     *
     * @return bool
     */
    protected function hasBiddingWar()
    {
        if (!$this->previous_bid) {
            return false;
        }

        return !empty(Rule::findLowestValidBidStep($this->previous_bid, $this->previous_bid->bid));
    }

    /**
     * There are 2 users with valid bids, we need to find out who has highest bid
     *
     * @return $this
     * @throws \Exception
     */
    protected function doBiddingWar()
    {
        $my_highest_bid       = Rule::findHighestValidBidStep($this, $this->previous_bid->bid);
        $previous_highest_bid = Rule::findHighestValidBidStep($this->previous_bid, $this->previous_bid->bid);

        if ($my_highest_bid > $previous_highest_bid) {
            $this->bid = Rule::findNextBidStep($this, $previous_highest_bid);
            $this->setBidValueInCurrency();
            $this->save();
//            $this->tryNotifySuccessfulBid();
            $this->previous_bid->tryNotifyOutbid();

            return $this;
        } else {
            $this->bid = $this->max_bid;
            $this->setBidValueInCurrency();
            $this->save();
        }

        // This bid lost, bump previous to the highest bid
        $winner            = new self;
        $winner->lot_id    = $this->lot_id;
        $winner->member_id = $this->previous_bid->member_id;
        $winner->max_bid   = $this->previous_bid->max_bid;
        $winner->bid       = Rule::findNextBidStep($this, $my_highest_bid);
        if ($winner->bid > $winner->max_bid) {
            $winner->bid = $winner->max_bid;
        }
        $winner->setBidValueInCurrency();
        $winner->save();
        throw new \Exception(__('auction::lots.bidding.outbid_by_max_bid'));
    }

    /**
     * Check if Bid is under Reserve price
     *
     * @return bool
     */
    protected function isBidUnderReservePrice()
    {
        if ($this->lot->reserve_price == 0) {
            return false;
        }

        return $this->max_bid < $this->lot->reserve_price;
    }

    /**
     * Check if this Bid breaches Reserve price
     *
     * @return bool
     */
    protected function isReserveBreachBid()
    {
        if ($this->lot->reserve_price == 0) {
            return false;
        }

        return $this->max_bid >= $this->lot->reserve_price
               && ($this->previous_bid->bid ?? 0) < $this->lot->reserve_price;
    }

    /**
     * Check if Anti Sniping should be triggered for the lot
     *
     * @return bool
     */
    protected function shouldTriggerAntiSniping()
    {
        if (config('auction.settings.anti_sniping_enabled') !== true) {
            return false;
        }
        
        if(Carbon::now()
                  ->diffInHours($this->lot->auction->original_ends_at) >= config('auction.settings.anti_sniping_max_overtime_hours')){
            return false;
        }

        return Carbon::now()
                     ->diffInMinutes($this->lot->ends_at) < config('auction.settings.anti_sniping_cutoff_time');
    }

}