<?php

/**
 * Brand Object Eloquent model
 *
 * @version 16/09/16
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */

namespace Mtc\Auction;

use Illuminate\Database\Eloquent\Model;
use Mtc\Core\Facades\Media;
use Mtc\Shop\Item;

/**
 * Brand Object Eloquent model
 *
 * @version 16/09/16
 * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class Brand extends Model
{
    /**
     * @var string Model table name
     */
    protected $table = 'auction_brands';

    /**
     * @var array|bool DateTime columns for object
     */
    public $timestamps = false;

    /**
     * @var array The attributes that are mass assignable.
     */
    protected $fillable = [
        'order',
        'name',
        'description',
        'hide',
        'image',
        'logo',
        'seo_title',
        'seo_keywords',
        'seo_description'
    ];

    /**
     * Scope - active()
     * Discards all hidden and deleted brands
     *
     * @param \Illuminate\Database\Eloquent\Builder $query Query builder object
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeActive($query)
    {
        return $query->where('hide', 0)
            ->where('deleted', 0);
    }

    /**
     * Scope - selectBasics()
     * Retrieves basic info about brand - id & name
     *
     * @param \Illuminate\Database\Eloquent\Builder $query Query builder object
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeSelectBasics($query)
    {
        return $query->select([
            'id',
            'name'
        ]);
    }

    /*
     * Items relationship.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function lots()
    {
        return $this->belongsToMany(Lot::class, 'auction_lot_brands', 'brand_id');
    }

    /**
     * @param string $size
     * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Contracts\Routing\UrlGenerator|string
     */
    public function getImageUrlAttribute(string $size = 'normal')
    {
        if (empty($this->attributes['image'])) {
            return '';
        }
        $folders = Media::getFolders('brand_images');
        return url($folders[$size]['path'] . '/' . $this->attributes['image']);
    }

    /**
     * @param string $size
     * @return \Illuminate\Contracts\Foundation\Application|\Illuminate\Contracts\Routing\UrlGenerator|string
     */

    public function getLogoUrlAttribute($size = 'normal')
    {
        if (empty($this->attributes['logo'])) {
            return '';
        }
        $folders = Media::getFolders('brand_logos');
        return url($folders[$size]['path'] . '/' . $this->attributes['logo']);
    }

    /**
     * Get all active brands ordered by order and name.
     * This function generates a URL for all returned brands
     *
     * @return array
     */
    public static function getAllBrands()
    {
        $query = Brand::active()
            ->selectBasics()
            ->orderBy('order', 'ASC')
            ->orderBy('name', 'ASC');

        return $query->get()
            ->map(function ($brand) {
                $brand['url'] = browse_url(0, $brand->id);
                return $brand;
            })
            ->toArray();
    }
}
