<?php

namespace Mtc\Auction\Drivers;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Arr;

trait DataImportable
{

    private $already_suggested_columns = [];

    public function __construct()
    {
        $required_attributes = [
            'model',
            'columns',
            'defaults',
        ];

        foreach ($required_attributes as $attribute) {
            if (!isset($this->$attribute)) {
                throw new \Exception(__CLASS__ . ' must have the \'' . $attribute . '\' attribute');
            }
        }
    }

    public static function getName()
    {
        return ucwords(str_replace('_', ' ', self::getSlug()));
    }

    public static function getSlug()
    {
        $model_name = (new self)->model;
        $model = new $model_name;

        if (!($model instanceof Model)) {
            throw new \Exception('$model must be an instance of ' . Model::class);
        }

        return $model->getTable();
    }

    public function getAvailableColumns()
    {
        return collect($this->getColumns())->map(function ($column) {
            return Arr::only($column, [
                'display_name',
                'attribute_name',
            ]);
        });
    }

    public function getSuggestedColumn($csv_column_name)
    {
        if(is_null($this->columns) || empty($this->columns)) {
            $this->columns = $this->getColumns();
        }

        $suggestion = collect($this->columns)
            ->filter(function ($column) use ($csv_column_name) {
                if (in_array($column['attribute_name'], $this->already_suggested_columns)) {
                    return false;
                }

                foreach ($column['match_suggestions'] as $match_suggestion) {
                    if (preg_match("/$match_suggestion/i", $csv_column_name)) {
                        $this->already_suggested_columns[] = $column['attribute_name'];
                        return true;
                    }
                }

                return false;
            })
            ->first();

        if(!is_null($suggestion)){
            return $suggestion['attribute_name'];
        }
        return '';
    }

    public function import($import_data)
    {
        $import_data = collect($import_data);

        foreach ($this->defaults as $column => $default) {
            if (!$import_data->has($column)) {
                $import_data->put($column, $default);
            }
        }

        // Map values as required
        $import_data = $import_data
            ->map(function ($import_line, $key) {
                return $this->mapData($import_line, $key);
            });

        // Group data by relations - root model, then each relationship of that model based on dot notation (only supports one relationship deep)
        $import_data = $import_data
            ->groupBy(function ($import_line, $key) {
                if (preg_match('/(.*)\..*/', $key, $matches)) {
                    return $matches[1];
                }
                return 'root';
            }, $preserveKeys = true)
            ->mapWithKeys(function ($import_line, $line_key) {
                $import_line = $import_line->mapWithKeys(function ($cell, $cell_key) use ($line_key) {
                    if ($line_key !== 'root') {
                        $cell_key = preg_replace('/^(.*\.)/', '', $cell_key);
                    }

                    return [$cell_key => $cell];
                })->toArray();

                return [$line_key => $import_line];
            });

        // Make sure there's a "root" key
        if (!$import_data->has('root')) {
            $import_data->put('root', []);
        }

        // Make sure the "root" key is first
        $import_data->prepend($import_data->pull('root'), 'root');

        // Map relation data as required
        $import_data = $import_data
            ->map(function ($relation_data, $relation) {
                return $this->mapRelationData($relation_data, $relation);
            });

        // Do the import
        $import_data->each(function ($import_data, $relation) use (&$model) {
            if ($relation === 'root') {
                $model = new $this->model;
                foreach ($import_data as $attribute_name => $value) {
                    if($attribute_name === 'Don\'t import'){
                        continue;
                    }
                    if($attribute_name === 'date_of_birth') {
                        if (\DateTime::createFromFormat('Y-m-d', $value) !== false) {
                            $value = Carbon::createFromFormat('Y-m-d', $value)
                                           ->format('Y-m-d');
                        } else {
                            $value = '';
                        }
                    }
                    $model->$attribute_name = $value;
                }
                $model->save();
            } else {
                $model_relation = $model->$relation()->newRelatedInstanceFor($model);
                foreach ($import_data as $attribute_name => $value) {
                    if($attribute_name === 'Don\'t import'){
                        continue;
                    }

                    $model_relation->$attribute_name = $value;
                }
                $model_relation->save();

            }
        });
    }

    public function mapData($import_line, $attribute_name)
    {
        return $import_line;
    }

    public function mapRelationData($relation_data, $relation)
    {
        return $relation_data;
    }

}